import { Component, inject, OnInit } from '@angular/core';
import { NGX_ECHARTS_CONFIG, NgxEchartsModule } from 'ngx-echarts';
import { DashboradService } from '../dashboard.service';
import { BaseComponent } from '../../base.component';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { FormsModule } from '@angular/forms';
import { distinctUntilChanged, Subject } from 'rxjs';
import { debounceTime } from 'rxjs/operators';
import { HttpResponse } from '@angular/common/http';
import { CommonOverViewChart } from '../model/commonOverViewChart';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-non-conformance-chart',
  imports: [
    NgxEchartsModule,
    MatDatepickerModule,
    FormsModule,
    PageHelpTextComponent,
    TranslateModule,
    MatIconModule,
    MatCardModule
  ],
  providers: [
    {
      provide: NGX_ECHARTS_CONFIG,
      useValue: {
        echarts: () => import('echarts'),
      },
    }
  ],
  templateUrl: './non-conformance-chart.component.html',
  styleUrl: './non-conformance-chart.component.scss'
})
export class NonConformanceChartComponent extends BaseComponent implements OnInit {
  dashboardService = inject(DashboradService);

  filterParameter$: Subject<string> = new Subject<string>();
  _FromDateFilter: Date | null = null;
  _ToDateFilter: Date | null = null;

  public get fromDateFilter(): Date | null {
    return this._FromDateFilter;
  }

  public set fromDateFilter(v: Date | null) {
    if (this._FromDateFilter !== v) {
      this._FromDateFilter = v;
      this.filterParameter$.next(`createdFromDate#${v ? v : ''}`);
    }
  }

  public get toDateFilter(): Date | null {
    return this._ToDateFilter;
  }

  public set toDateFilter(v: Date | null) {
    if (this._ToDateFilter !== v) {
      this._ToDateFilter = v;
      this.filterParameter$.next(`createdToDate#${v ? v : ''}`);
    }
  }

  ngOnInit(): void {
    this.getNonConformanceChart();

    this.filterParameter$.pipe(
      distinctUntilChanged(),
      debounceTime(1000),
    ).subscribe((filter) => {
      const [key, value] = filter.split('#');
      if (key === 'createdFromDate') {
        this._FromDateFilter = value ? new Date(value) : null;
      }
      if (key === 'createdToDate') {
        this._ToDateFilter = value ? new Date(value) : null;
      }
      this.getNonConformanceChart(this.fromDateFilter ?? undefined, this.toDateFilter ?? undefined);
    });
  }

  clearOrderDates() {
    this.fromDateFilter = null;
    this.toDateFilter = null;
  }

  getNonConformanceChart(fromDate?: Date, toDate?: Date) {
    this.sub$.sink = this.dashboardService.getNonConformanceChart(fromDate, toDate).subscribe({
      next: (data) => {
        const response = data as HttpResponse<CommonOverViewChart[]>;
        if (response && response.body) {
          const statusOrder = ["OPEN", "INPROGRESS", "CLOSED", "REJECTED"];
          const chartData = statusOrder.map(status => {
            const found = response.body?.find((item: CommonOverViewChart) => item.name === status);
            return found ? found.count : 0;
          });
          this.ncStatusChartOptions = {
            ...this.ncStatusChartOptions,
            series: [
              {
                ...this.ncStatusChartOptions.series[0],
                data: chartData,
              }
            ]
          };
        }
      },
    });
  }

  ncStatusChartOptions: any = {
    title: {
      text: '',
      left: 'center'
    },
    tooltip: {
      trigger: 'axis',
      axisPointer: {
        type: 'shadow'
      }
    },
    xAxis: {
      type: 'category',
      data: ['OPEN', 'INPROGRESS', 'CLOSED', 'REJECTED'].map(item => this.translationService.getValue(item)),
      axisLabel: {
        rotate: 15
      }
    },
    yAxis: {
      type: 'value',
      min: 0,
      max: 30,
      interval: 5,
      axisLabel: {
        formatter: (value: number) => {
          return [5, 10, 15, 20, 25, 30].includes(value)
            ? value.toString()
            : '';
        },
      },
    },
    series: [
      {
        data: [] as number[],
        type: 'bar',
        itemStyle: {
          color: function (params: any) {
            const statusColorMap: Record<string, string> = {
              'Open': '#007bff',
              'In Progress': '#ffc107',
              'Closed': '#28a745',
              'Rejected': '#dc3545'
            };
            return statusColorMap[params.name] || '#888';
          }
        },
        label: {
          show: true,
          position: 'top'
        }
      }
    ]
  };
}
