import { Component, Inject, inject, OnInit, Optional } from '@angular/core';
import { MatCardModule } from '@angular/material/card';
import {
  FormBuilder,
  FormGroup,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { DepartmentStore } from '../../audit-template/department/store/department.store';
import { MatSelectModule } from '@angular/material/select';
import { ActivatedRoute, Router, RouterLink } from '@angular/router';
import { BaseComponent } from '../../base.component';
import { CommonService } from '@core/services/common.service';
import { NON_CONFORMANCE_STATUS } from '../../non-conformance/model/non-conformance-status-enum';
import { NON_CONFORMANCE_SEVERITY } from '../../non-conformance/model/non-conformance-severity-enum';
import { NonConformanceStore } from '../store/non-conformance-store';
import { NonConformance } from '../../non-conformance/model/non-conformance';
import { toObservable } from '@angular/core/rxjs-interop';
import { NonConformanceService } from '../non-conformance.service';
import { MatIconModule } from '@angular/material/icon';
import { TextEditorComponent } from '@shared/text-editor/text-editor.component';
import { UserStore } from '../../user/store/user.store';
import { AuditService } from '../../audit/audit.service';
import { IdName } from '@core/domain-classes/id-name';
import { CommonError } from '@core/error-handler/common-error';

import { ToastrService } from '@core/services/toastr-service';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { MatButtonModule } from '@angular/material/button';
import { NgClass } from '@angular/common';
import { MAT_DIALOG_DATA, MatDialog, MatDialogRef } from '@angular/material/dialog';
import { ManageDepartmentComponent } from '../../audit-template/department/manage-department/manage-department.component';
import { HasClaimDirective } from '@shared/has-claim.directive';

@Component({
  selector: 'app-manage-non-conformance',
  imports: [
    PageHelpTextComponent,
    MatCardModule,
    ReactiveFormsModule,
    MatSelectModule,
    MatIconModule,
    TextEditorComponent,
    TranslateModule,
    MatButtonModule,
    NgClass,
    HasClaimDirective
  ],
  templateUrl: './manage-non-conformance.component.html',
  styleUrl: './manage-non-conformance.component.scss',
})
export class ManageNonConformanceComponent
  extends BaseComponent
  implements OnInit {
  nonConformanceForm: FormGroup;
  fb = inject(FormBuilder);
  departmentStore = inject(DepartmentStore);
  commonService = inject(CommonService);
  nonConformanceStore = inject(NonConformanceStore);
  nonConformanceService = inject(NonConformanceService);
  toastrService = inject(ToastrService);
  auditService = inject(AuditService);
  auditList: IdName[] = [];
  userStore = inject(UserStore);
  isEditMode: boolean = false;
  saving: boolean = false;
  router = inject(Router);
  route = inject(ActivatedRoute);
  dialog = inject(MatDialog);
  nonConformanceId: string = '';
  nonConformance: NonConformance;
  isDialog: boolean = false;
  shouldDisableAuditId: boolean = false;
  
  statusList = Object.keys(NON_CONFORMANCE_STATUS)
    .filter((key) => !isNaN(Number(NON_CONFORMANCE_STATUS[key as any])))
    .map((key) => ({
      label: key,
      value: NON_CONFORMANCE_STATUS[key as keyof typeof NON_CONFORMANCE_STATUS],
    }));
  severityList = Object.keys(NON_CONFORMANCE_SEVERITY)
    .filter((key) => !isNaN(Number(NON_CONFORMANCE_SEVERITY[key as any])))
    .map((key) => ({
      name: key,
      id: NON_CONFORMANCE_SEVERITY[
        key as keyof typeof NON_CONFORMANCE_SEVERITY
      ],
    }));

  constructor(
    @Optional() private dialogRef: MatDialogRef<ManageNonConformanceComponent>,
    @Optional() @Inject(MAT_DIALOG_DATA) public data: { event: { auditId: string; auditResponseId: string; } },
  ) {
    super();
    this.subscribeIsAddorUpdate();
  }

  ngOnInit(): void {
    this.createNonConformanceForm();

    if (this.data?.event) {
      this.nonConformanceForm.patchValue({
        auditId: this.data.event.auditId,
        auditResponseId: this.data.event.auditResponseId,
      });
      this.nonConformanceForm.get('auditId')?.disable();
      this.isDialog = true;
    }

    this.auditService
      .getAuditsForDropdown()
      .subscribe((result: IdName | CommonError) => {
        if (Array.isArray(result)) {
          this.auditList = result;
        } else {
          this.auditList = [];
        }

        this.route.data.subscribe((data) => {
          const nc: NonConformance = data['nonConformanceDetail'];
          this.nonConformance = nc;
          this.nonConformanceId = nc?.id;
          this.isEditMode = !!this.nonConformanceId;

          if (this.isEditMode && nc) {
            this.nonConformanceForm.patchValue({
              ...nc,
              status: nc.status,
              severity: nc.severity,
              auditId: nc.auditId || '',
              reportedById: nc.reportedById || '',
            });
            this.nonConformanceForm.get('auditId')?.disable();
            this.shouldDisableAuditId = true;
          } else {
            this.getNonConformanceNUmber();

            if (this.data.event) {
              const matchingAudit = this.auditList.find(
                (a) => a.id === this.data.event.auditId
              );
              if (!matchingAudit) {
                this.toastrService.error(
                  this.translationService.getValue('AUDIT_NOT_FOUND')
                );
              } else {
                this.nonConformanceForm.patchValue({
                  auditId: matchingAudit.id,
                });
                this.nonConformanceForm.get('auditId')?.disable();
                this.shouldDisableAuditId = true;
              }
            }
            if (this.data.event) {
              this.nonConformanceForm.patchValue({
                auditResponseId: this.data.event.auditResponseId,
              });
            }
          }
        });
      });
  }

  getNonConformanceNUmber() {
    this.sub$.sink = this.nonConformanceService
      .getNonConformanceNumber()
      .subscribe((nc) => {
        const nonConformance = nc as NonConformance;
        if (nonConformance) {
          this.nonConformanceForm.patchValue({
            ncNumber: nonConformance.ncNumber,
          });
        }
      });
  }

  getAuditForDropdown(): Promise<void> {
    return new Promise((resolve) => {
      this.auditService
        .getAuditsForDropdown()
        .subscribe((result: IdName | CommonError) => {
          if (Array.isArray(result)) {
            this.auditList = result;
          } else {
            this.auditList = [];
          }
          resolve();
        });
    });
  }

  createNonConformanceForm() {
    this.nonConformanceForm = this.fb.group({
      id: [''],
      ncNumber: ['', [Validators.required]],
      title: ['', [Validators.required]],
      reportedById: ['', [Validators.required]],
      departmentId: [''],
      status: [null, [Validators.required]],
      severity: [null, [Validators.required]],
      auditId: [''],
      auditResponseId: [''],
      description: [''],
    });
  }

  onSubmit() {
    if (!this.nonConformanceForm.valid) {
      this.nonConformanceForm.markAllAsTouched();
      this.toastrService.error(this.translationService.getValue('PLEASE_FIX_THE_FORM_ERRORS_BEFORE_SUBMITTING'));
      return;
    }
    const formValue: NonConformance = this.nonConformanceForm.getRawValue();

    if (this.isEditMode) {
      this.nonConformanceStore.updateNonConformance(formValue);
    } else {
      this.nonConformanceStore.addNonConformance(formValue);
    }
  }

  subscribeIsAddorUpdate() {
    this.sub$.sink = toObservable(this.nonConformanceStore.isAddUpdate).subscribe((flag) => {
      if (!flag) return;

      this.nonConformanceStore.resetflag();

      if (this.data !== null) {
        this.dialogRef.close(true);
        return;
      }

      this.router.navigate(['/nc/conformances']);;
    });
  }

  addDepartment(): void {
    const dialogRef = this.dialog.open(ManageDepartmentComponent, {
      width: '500px',
    });
    dialogRef.afterClosed().subscribe((result) => {
      if (result) {
        this.nonConformanceForm.patchValue({
          departmentId: result.id,
        });
      }
    });
  }

  closeDialog(): void {
    if (this.isDialog) {
      this.dialogRef.close(true);
      return;
    }
    this.router.navigate(['/nc/conformances']);
  }
}
