import {
  AfterViewInit,
  Component,
  inject,
  OnInit,
  ViewChild,
} from '@angular/core';
import { MatPaginator } from '@angular/material/paginator';
import { MatSort, MatSortModule } from '@angular/material/sort';
import { MatTableModule } from '@angular/material/table';
import { ActivatedRoute, RouterLink } from '@angular/router';
import { BaseComponent } from '../../base.component';
import { NonConformanceStore } from '../store/non-conformance-store';
import { PageSizeOption } from '@core/utils/global-config';
import { NON_CONFORMANCE_STATUS } from '../model/non-conformance-status-enum';
import { NON_CONFORMANCE_SEVERITY } from '../model/non-conformance-severity-enum';
import { CommonService } from '@core/services/common.service';
import { MatSelectModule } from '@angular/material/select';
import { DepartmentStore } from '../../audit-template/department/store/department.store';
import { debounceTime, distinctUntilChanged, merge, Subject, tap } from 'rxjs';
import { FormsModule } from '@angular/forms';
import { MatMenuModule } from '@angular/material/menu';
import { MatIconModule } from '@angular/material/icon';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import {
  trigger,
  state,
  style,
  transition,
  animate,
} from '@angular/animations';
import { NonConformanceResponse } from '../model/non-conformance-response';
import { UserStore } from '../../user/store/user.store';
import { NonConformanceService } from '../non-conformance.service';
import { IdName } from '@core/domain-classes/id-name';
import { CommonError } from '@core/error-handler/common-error';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { LimitToPipe } from '@shared/pipes/limit-to.pipe';
import { NonConformanceResponseListComponent } from '../non-conformance-responses/non-conformance-response-list/non-conformance-response-list.component';
import { NonConformanceStatusPipe } from '../pipes/non-conformance-status.pipe';
import { NgClass } from '@angular/common';
import { NonConformanceSeverityPipe } from '../pipes/non-conformance-severity.pipe';
import { MatButtonModule } from '@angular/material/button';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-non-conformance-list',
  imports: [
    MatTableModule,
    MatSortModule,
    MatPaginator,
    RouterLink,
    MatSelectModule,
    FormsModule,
    MatMenuModule,
    MatIconModule,
    NonConformanceResponseListComponent,
    PageHelpTextComponent,
    HasClaimDirective,
    TranslateModule,
    LimitToPipe,
    NonConformanceStatusPipe,
    NonConformanceSeverityPipe,
    NgClass,
    MatButtonModule,
    MatCardModule
  ],
  templateUrl: './non-conformance-list.component.html',
  styleUrl: './non-conformance-list.component.scss',
  animations: [
    trigger('detailExpand', [
      state('collapsed', style({ height: '0px', minHeight: '0' })),
      state('expanded', style({ height: '*' })),
      transition(
        'expanded <=> collapsed',
        animate('225ms cubic-bezier(0.4, 0.0, 0.2, 1)')
      ),
    ]),
  ],
})
export class NonConformanceListComponent
  extends BaseComponent
  implements OnInit, AfterViewInit {
  displayedColumns: string[] = [
    'action',
    'ncNumber',
    'title',
    'reportedBy',
    'severity',
    'status',
    'department',
    'audit',
  ];
  displayedColumnSecondary: string[] = [
    'action-search',
    'ncNumber-search',
    'title-search',
    'reportedBy-search',
    'severity-search',
    'status-search',
    'department-search',
    'audit-search',
  ];
  footerToDisplayed: string[] = ['footer'];
  userStore = inject(UserStore);
  nonConformanceStore = inject(NonConformanceStore);
  pageOption = PageSizeOption;
  commonService = inject(CommonService);
  route = inject(ActivatedRoute);
  nonConformanceService = inject(NonConformanceService);
  departmentStore = inject(DepartmentStore);
  @ViewChild(MatPaginator) paginator!: MatPaginator;
  @ViewChild(MatSort) sort!: MatSort;
  commandDialogService = inject(CommonDialogService);
  auditList: IdName[] = [];
  expandedElement!: NonConformanceResponse | null;
  statusList = Object.keys(NON_CONFORMANCE_STATUS)
    .filter((key) => !isNaN(Number(NON_CONFORMANCE_STATUS[key as any])))
    .map((key) => ({
      status: key,
      id: NON_CONFORMANCE_STATUS[key as keyof typeof NON_CONFORMANCE_STATUS],
    }));
  severityList = Object.keys(NON_CONFORMANCE_SEVERITY)
    .filter((key) => !isNaN(Number(NON_CONFORMANCE_SEVERITY[key as any])))
    .map((key) => ({
      name: key,
      id: NON_CONFORMANCE_SEVERITY[
        key as keyof typeof NON_CONFORMANCE_SEVERITY
      ],
    }));
  nonConformanceFilterParameter = {
    ...this.nonConformanceStore.filterParameters(),
  };
  filterParameter$: Subject<string> = new Subject<string>();
  _filterNcNumber = '';
  _filterReportedBy = '';
  _filterAudit = '';
  _filterSeverity = '';
  _filterStatus = '';
  _filterDepartment = '';
  _filterNcTitle = '';

  get filterNcNumber(): string {
    return this._filterNcNumber;
  }

  set filterNcNumber(value: string) {
    if (this._filterNcNumber != value) {
      this._filterNcNumber = value;
      this.nonConformanceFilterParameter.ncNumber = value;
      const filterNcNumber = `filterNcNumber#${value}`;
      this.filterParameter$.next(filterNcNumber);
    }
  }

  get filterReportedBy(): string {
    return this._filterReportedBy;
  }

  set filterReportedBy(value: string) {
    if (this._filterReportedBy != value) {
      this._filterReportedBy = value;
      this.nonConformanceFilterParameter.reportedById = value;
      const filterReportedBy = `filterReportedBy#${value}`;
      this.filterParameter$.next(filterReportedBy);
    }
  }

  get filterAudit(): string {
    return this._filterAudit;
  }

  set filterAudit(value: string) {
    if (this._filterAudit != value) {
      this._filterAudit = value;
      this.nonConformanceFilterParameter.auditId = value;
      const filterAudit = `filterAudit#${value}`;
      this.filterParameter$.next(filterAudit);
    }
  }

  get filterSeverity(): string {
    return this._filterSeverity;
  }

  set filterSeverity(value: string) {
    if (this._filterSeverity !== value) {
      this._filterSeverity = value;
      this.nonConformanceFilterParameter.severity = value;
      const filterSeverity = `filterSeverity#${value}`;
      this.filterParameter$.next(filterSeverity);
    }
  }

  get filterStatus(): string {
    return this._filterStatus;
  }

  set filterStatus(value: string) {
    if (this._filterStatus !== value) {
      this._filterStatus = value;
      this.nonConformanceFilterParameter.status = value;
      const filterStatus = `filterStatus#${value}`;
      this.filterParameter$.next(filterStatus);
    }
  }

  get filterDepartment(): string {
    return this._filterDepartment;
  }

  set filterDepartment(value: string) {
    if (this._filterDepartment !== value) {
      this._filterDepartment = value;
      this.nonConformanceFilterParameter.departmentId = value;
      const filterDepartment = `filterDepartment#${value}`;
      this.filterParameter$.next(filterDepartment);
    }
  }

  get filterNcTitle(): string {
    return this._filterNcTitle;
  }

  set filterNcTitle(value: string) {
    if (this._filterNcTitle !== value) {
      this._filterNcTitle = value;
      this.nonConformanceFilterParameter.title = value;
      const filterNcTitle = `filterNcTitle#${value}`;
      this.filterParameter$.next(filterNcTitle);
    }
  }

  ngOnInit(): void {
    this.getAllAudits();
    this.route.queryParamMap.subscribe((queryParams) => {
      const auditId = queryParams.get('auditId');
      if (auditId) {
        const auditCheckInterval = setInterval(() => {
          if (this.auditList.length > 0) {
            clearInterval(auditCheckInterval);
            this.filterAudit = auditId;
            this.nonConformanceFilterParameter.auditId = auditId;
          }
        }, 100);
      }
    });
    this.sub$.sink = this.filterParameter$
      .pipe(debounceTime(1000), distinctUntilChanged())
      .subscribe((filter: string) => {
        const [key, value] = filter.split('#');
        if (key === 'filterNcNumber') {
          this.nonConformanceFilterParameter.ncNumber = value;
        } else if (key === 'filterReportedBy') {
          this.nonConformanceFilterParameter.reportedById = value;
        } else if (key === 'filterAudit') {
          this.nonConformanceFilterParameter.auditId = value;
        } else if (key === 'filterSeverity') {
          this.nonConformanceFilterParameter.severity = value;
        } else if (key === 'filterStatus') {
          this.nonConformanceFilterParameter.status = value;
        } else if (key === 'filterDepartment') {
          this.nonConformanceFilterParameter.departmentId = value;
        } else if (key === 'filterNcTitle') {
          this.nonConformanceFilterParameter.title = value;
        }
        this.nonConformanceStore.loadByQuery(
          this.nonConformanceFilterParameter
        );
      });
  }

  getNCResponse(ncResponse: NonConformanceResponse) {
    if (ncResponse) {
      this.expandedElement =
        this.expandedElement === ncResponse ? null : ncResponse;
    }
  }

  refresh() {
    this.nonConformanceStore.loadByQuery(this.nonConformanceFilterParameter);
  }

  getAllAudits() {
    this.nonConformanceService.getAllAudits().subscribe({
      next: (result: IdName[] | CommonError) => {
        if (Array.isArray(result)) {
          this.auditList = result;
        } else {
          this.auditList = [];
        }
      },
    });
  }

  deleteNonConformance(id: string) {
    this.commandDialogService
      .deleteConfirmtionDialog(
        this.translationService.getValue(
          'ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_NON_CONFORMANCE'
        )
      )
      .subscribe({
        next: (result: boolean) => {
          if (result) {
            this.nonConformanceStore.deleteNonConformanceById(id);
          }
        },
      });
  }

  ngAfterViewInit(): void {
    this.sub$.sink = this.sort.sortChange.subscribe(() => {
      this.paginator.pageIndex = 0;
    });
    this.sub$.sink = merge(this.sort.sortChange, this.paginator.page)
      .pipe(
        tap(() => {
          this.nonConformanceFilterParameter.skip =
            this.paginator.pageIndex * this.paginator.pageSize;
          this.nonConformanceFilterParameter.pageSize = this.paginator.pageSize;
          this.nonConformanceFilterParameter.orderBy =
            this.sort.active + ' ' + this.sort.direction;
          this.nonConformanceStore.loadByQuery(
            this.nonConformanceFilterParameter
          );
        })
      )
      .subscribe();
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.nonConformanceStore.nonConformanceList().indexOf(row);
  }
}
