import { Component, inject, OnInit, ViewChild } from '@angular/core';
import { MatPaginator } from '@angular/material/paginator';
import { MatSort, MatSortModule } from '@angular/material/sort';
import { ActivatedRoute, RouterModule } from '@angular/router';
import { IdName } from '@core/domain-classes/id-name';
import { CommonError } from '@core/error-handler/common-error';
import { PageSizeOption } from '@core/utils/global-config';
import { Subject, debounceTime, distinctUntilChanged, merge, tap } from 'rxjs';
import { DepartmentStore } from '../../audit-template/department/store/department.store';
import { UserStore } from '../../user/store/user.store';
import { NON_CONFORMANCE_SEVERITY } from '../model/non-conformance-severity-enum';
import { NON_CONFORMANCE_STATUS } from '../model/non-conformance-status-enum';
import { NonConformanceService } from '../non-conformance.service';
import { BaseComponent } from '../../base.component';
import { NonConformanceResource } from '../model/non-conformance-resource';
import { NonConformanceLogService } from './non-conformance-log.service';
import { HttpResponse } from '@angular/common/http';
import { NonConformance } from '../model/non-conformance';
import { MatTableModule } from '@angular/material/table';
import { MatSelectModule } from '@angular/material/select';
import { FormsModule } from '@angular/forms';
import { NON_CONFORMANCE_LOG_STATUS } from '../model/non-conformance-log-status.enum';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { LimitToPipe } from '@shared/pipes/limit-to.pipe';
import { NonConformanceLogStatusPipe } from '../pipes/non-conformance-log-status.pipe';
import { NonConformanceSeverityPipe } from '../pipes/non-conformance-severity.pipe';
import { NonConformanceStatusPipe } from '../pipes/non-conformance-status.pipe';
import { MatCardModule } from '@angular/material/card';
import { NgClass } from '@angular/common';

@Component({
  selector: 'app-non-conformance-log-list',
  imports: [
    PageHelpTextComponent,
    MatTableModule,
    MatPaginator,
    MatSortModule,
    MatSelectModule,
    FormsModule,
    RouterModule,
    TranslateModule,
    LimitToPipe,
    NonConformanceLogStatusPipe,
    NonConformanceStatusPipe,
    NonConformanceSeverityPipe,
    MatCardModule,
    NgClass
  ],
  templateUrl: './non-conformance-log-list.component.html',
  styleUrl: './non-conformance-log-list.component.scss'
})
export class NonConformanceLogListComponent extends BaseComponent implements OnInit {
  displayedColumns: string[] = [
    'ncNumber',
    'title',
    'reportedBy',
    'severity',
    'status',
    'department',
    'audit',
    'logStatus',
  ];
  displayedColumnSecondary: string[] = [
    'ncNumber-search',
    'title-search',
    'reportedBy-search',
    'severity-search',
    'status-search',
    'department-search',
    'audit-search',
    'logStatus-search',
  ];
  footerToDisplayed: string[] = ['footer'];
  userStore = inject(UserStore);
  pageOption = PageSizeOption;
  route = inject(ActivatedRoute);
  nonConformanceService = inject(NonConformanceService);
  nonConformanceLogService = inject(NonConformanceLogService);
  departmentStore = inject(DepartmentStore);
  auditList: IdName[] = [];
  nonConformanceList: NonConformance[] = [];

  @ViewChild(MatPaginator) paginator!: MatPaginator;
  @ViewChild(MatSort) sort!: MatSort;

  statusList = Object.keys(NON_CONFORMANCE_STATUS)
    .filter((key) => !isNaN(Number(NON_CONFORMANCE_STATUS[key as any])))
    .map((key) => ({
      status: key,
      id: NON_CONFORMANCE_STATUS[key as keyof typeof NON_CONFORMANCE_STATUS],
    }));
  severityList = Object.keys(NON_CONFORMANCE_SEVERITY)
    .filter((key) => !isNaN(Number(NON_CONFORMANCE_SEVERITY[key as any])))
    .map((key) => ({
      name: key,
      id: NON_CONFORMANCE_SEVERITY[
        key as keyof typeof NON_CONFORMANCE_SEVERITY
      ],
    }));
  logStatus = Object.keys(NON_CONFORMANCE_LOG_STATUS)
    .filter((key) => !isNaN(Number(NON_CONFORMANCE_LOG_STATUS[key as any])))
    .map((key) => ({
      name: key,
      id: NON_CONFORMANCE_LOG_STATUS[key as keyof typeof NON_CONFORMANCE_LOG_STATUS],
    }));

  nonConformanceFilterParameter: NonConformanceResource = {
    orderBy: 'ncNumber desc',
    pageSize: 10,
    skip: 0,
    totalCount: 0,
    ncNumber: '',
    reportedById: '',
    severity: '',
    status: '',
    departmentId: '',
    auditId: '',
    title: '',
    logStatus: '',
  };
  filterParameter$: Subject<string> = new Subject<string>();
  _filterNcNumber = '';
  _filterReportedBy = '';
  _filterAudit = '';
  _filterSeverity = '';
  _filterStatus = '';
  _filterDepartment = '';
  _filterNcTitle = '';
  _filterLogStatus = '';

  get filterNcNumber(): string {
    return this._filterNcNumber;
  }

  set filterNcNumber(value: string) {
    if (this._filterNcNumber != value) {
      this._filterNcNumber = value;
      this.nonConformanceFilterParameter.ncNumber = value;
      const filterNcNumber = `filterNcNumber#${value}`;
      this.filterParameter$.next(filterNcNumber);
    }
  }

  get filterReportedBy(): string {
    return this._filterReportedBy;
  }

  set filterReportedBy(value: string) {
    if (this._filterReportedBy != value) {
      this._filterReportedBy = value;
      this.nonConformanceFilterParameter.reportedById = value;
      const filterReportedBy = `filterReportedBy#${value}`;
      this.filterParameter$.next(filterReportedBy);
    }
  }

  get filterAudit(): string {
    return this._filterAudit;
  }

  set filterAudit(value: string) {
    if (this._filterAudit != value) {
      this._filterAudit = value;
      this.nonConformanceFilterParameter.auditId = value;
      const filterAudit = `filterAudit#${value}`;
      this.filterParameter$.next(filterAudit);
    }
  }

  get filterSeverity(): string {
    return this._filterSeverity;
  }

  set filterSeverity(value: string) {
    if (this._filterSeverity !== value) {
      this._filterSeverity = value;
      this.nonConformanceFilterParameter.severity = value;
      const filterSeverity = `filterSeverity#${value}`;
      this.filterParameter$.next(filterSeverity);
    }
  }

  get filterStatus(): string {
    return this._filterStatus;
  }

  set filterStatus(value: string) {
    if (this._filterStatus !== value) {
      this._filterStatus = value;
      this.nonConformanceFilterParameter.status = value;
      const filterStatus = `filterStatus#${value}`;
      this.filterParameter$.next(filterStatus);
    }
  }

  get filterDepartment(): string {
    return this._filterDepartment;
  }

  set filterDepartment(value: string) {
    if (this._filterDepartment !== value) {
      this._filterDepartment = value;
      this.nonConformanceFilterParameter.departmentId = value;
      const filterDepartment = `filterDepartment#${value}`;
      this.filterParameter$.next(filterDepartment);
    }
  }

  get filterNcTitle(): string {
    return this._filterNcTitle;
  }

  set filterNcTitle(value: string) {
    if (this._filterNcTitle !== value) {
      this._filterNcTitle = value;
      this.nonConformanceFilterParameter.title = value;
      const filterNcTitle = `filterNcTitle#${value}`;
      this.filterParameter$.next(filterNcTitle);
    }
  }

  get filterLogStatus(): string {
    return this._filterLogStatus;
  }

  set filterLogStatus(value: string) {
    if (this._filterLogStatus !== value) {
      this._filterLogStatus = value;
      this.nonConformanceFilterParameter.logStatus = value;
      const filterLogStatus = `filterLogStatus#${value}`;
      this.filterParameter$.next(filterLogStatus);
    }
  }

  ngOnInit(): void {
    this.getAllAudits();
    this.refresh();

    this.route.queryParamMap.subscribe((queryParams) => {
      const auditId = queryParams.get('auditId');
      if (auditId) {
        const auditCheckInterval = setInterval(() => {
          if (this.auditList.length > 0) {
            clearInterval(auditCheckInterval);
            this.filterAudit = auditId;
            this.nonConformanceFilterParameter.auditId = auditId;
            this.refresh();
          }
        }, 100);
      }
    });
    this.sub$.sink = this.filterParameter$
      .pipe(debounceTime(1000), distinctUntilChanged())
      .subscribe((filter: string) => {
        const [key, value] = filter.split('#');
        if (key === 'filterNcNumber') {
          this.nonConformanceFilterParameter.ncNumber = value;
        } else if (key === 'filterReportedBy') {
          this.nonConformanceFilterParameter.reportedById = value;
        } else if (key === 'filterAudit') {
          this.nonConformanceFilterParameter.auditId = value;
        } else if (key === 'filterSeverity') {
          this.nonConformanceFilterParameter.severity = value;
        } else if (key === 'filterStatus') {
          this.nonConformanceFilterParameter.status = value;
        } else if (key === 'filterDepartment') {
          this.nonConformanceFilterParameter.departmentId = value;
        } else if (key === 'filterNcTitle') {
          this.nonConformanceFilterParameter.title = value;
        } else if (key === 'filterLogStatus') {
          this.nonConformanceFilterParameter.logStatus = value;
        }
        this.refresh();
      });
  }

  refresh() {
    this.loadByQuery(this.nonConformanceFilterParameter);
  }

  getAllAudits() {
    this.nonConformanceService.getAllAudits().subscribe({
      next: (result: IdName[] | CommonError) => {
        if (Array.isArray(result)) {
          this.auditList = result;
        } else {
          this.auditList = [];
        }
      },
    });
  }

  ngAfterViewInit(): void {
    this.sub$.sink = this.sort.sortChange.subscribe(() => {
      this.paginator.pageIndex = 0;
    });
    this.sub$.sink = merge(this.sort.sortChange, this.paginator.page)
      .pipe(
        tap(() => {
          this.nonConformanceFilterParameter.skip =
            this.paginator.pageIndex * this.paginator.pageSize;
          this.nonConformanceFilterParameter.pageSize = this.paginator.pageSize;
          this.nonConformanceFilterParameter.orderBy =
            this.sort.active + ' ' + this.sort.direction;
          this.refresh();
        })
      )
      .subscribe();
  }

  loadByQuery(filter: NonConformanceResource) {
    this.sub$.sink = this.nonConformanceLogService.getNonConformanceLog(filter).subscribe({
      next: (httpResponse: HttpResponse<NonConformance[]>) => {
        if (httpResponse && httpResponse.body) {
          this.nonConformanceList = [...httpResponse.body as NonConformance[]];
        }
        const paginationHeader = httpResponse.headers.get('x-pagination');
        if (paginationHeader) {
          const pagination = JSON.parse(paginationHeader);
          this.nonConformanceFilterParameter = { ...filter, totalCount: pagination.totalCount };
        }
      }
    });
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.nonConformanceList.indexOf(row);
  }
}
