import { computed, inject } from "@angular/core";
import { NonConformanceResponseType } from "../../../non-conformance/model/non-conformance-response-type";
import { CommonError } from "@core/error-handler/common-error";
import { patchState, signalStore, withComputed, withHooks, withMethods, withState } from "@ngrx/signals";
import { ToastrService } from '@core/services/toastr-service';
import { TranslationService } from "@core/services/translation.service";
import { rxMethod } from "@ngrx/signals/rxjs-interop";
import { debounceTime, distinctUntilChanged, pipe, switchMap, tap } from "rxjs";
import { tapResponse } from "@ngrx/operators";
import { toObservable } from "@angular/core/rxjs-interop";
import { NonConformanceResponseTypeService } from "../non-conformance-response-type.service";

type NonConformanceResponseTypeState = {
  nonConformanceResponseTypes: NonConformanceResponseType[];
  nonConformanceResponseType: NonConformanceResponseType | null;
  searchQuery: string;
  loadList: boolean;
  isAddUpdate: boolean;
};

export const initialNonConformanceResponseTypeState: NonConformanceResponseTypeState = {
  nonConformanceResponseTypes: [],
  nonConformanceResponseType: null,
  searchQuery: '',
  loadList: true,
  isAddUpdate: false,
};

export const NonConformanceResponseTypeStore = signalStore(
  { providedIn: 'root' },
  withState(initialNonConformanceResponseTypeState),
  withComputed(({ nonConformanceResponseTypes, searchQuery }) => ({
    filteredNonConformanceResponseTypes: computed(() => {
      const filter = searchQuery().toLowerCase();
      return nonConformanceResponseTypes()
        .filter(type => type.name.toLowerCase().includes(filter));
    })
  })),
  withMethods(
    (
      store,
      nonConformanceResponseTypeService = inject(NonConformanceResponseTypeService),
      toastrService = inject(ToastrService),
      translationService = inject(TranslationService),
    ) => ({
      setNameFilter(value: string) {
        patchState(store, { searchQuery: value });
      },
      loadNonConformanceResponseTypes: rxMethod<void>(
        pipe(
          debounceTime(300),
          tap(() => patchState(store, { nonConformanceResponseTypes: [] })),
          switchMap(() =>
            nonConformanceResponseTypeService.getNonConformanceResponseType().pipe(
              tapResponse({
                next: (responseTypes: NonConformanceResponseType[]) => {
                  patchState(store, {
                    nonConformanceResponseTypes: [...responseTypes],
                    loadList: false,
                  });
                },
                error: (err) => {
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      addNonConformanceResponseType: rxMethod<NonConformanceResponseType>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { loadList: false })),
          switchMap((type: NonConformanceResponseType) =>
            nonConformanceResponseTypeService.addNonConformanceResponseType(type).pipe(
              tapResponse({
                next: (response) => {
                  toastrService.success(
                    translationService.getValue('NON_CONFORMANCE_RESPONSE_TYPE_CREATED_SUCCESSFULLY')
                  );
                  patchState(store, {
                    loadList: true,
                    isAddUpdate: true,
                    nonConformanceResponseType: response as NonConformanceResponseType
                  });
                },
                error: (err: CommonError) => {
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      updateNonConformanceResponseType: rxMethod<NonConformanceResponseType>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { loadList: false })),
          switchMap((type: NonConformanceResponseType) =>
            nonConformanceResponseTypeService.updateNonConformanceResponseType(type).pipe(
              tapResponse({
                next: (response) => {
                  const updatedType: NonConformanceResponseType = response as NonConformanceResponseType;
                  toastrService.success(
                    translationService.getValue('NON_CONFORMANCE_RESPONSE_TYPE_UPDATED_SUCCESSFULLY')
                  );
                  patchState(store, {
                    nonConformanceResponseTypes: store.nonConformanceResponseTypes().map((t) => {
                      return updatedType.id === t.id ? updatedType : t;
                    }),
                    isAddUpdate: true,
                    nonConformanceResponseType: updatedType
                  });
                },
                error: (err: CommonError) => {
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      deleteNonConformanceResponseTypeById: rxMethod<string>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { loadList: false })),
          switchMap((typeId: string) =>
            nonConformanceResponseTypeService.deleteNonConformanceResponseType(typeId).pipe(
              tapResponse({
                next: () => {
                  toastrService.success(
                    translationService.getValue('NON_CONFORMANCE_RESPONSE_TYPE_DELETED_SUCCESSFULLY')
                  );
                  patchState(store, {
                    nonConformanceResponseTypes: store.nonConformanceResponseTypes().filter((t) => t.id !== typeId),
                  });
                },
                error: (err: CommonError) => {
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      resetFlag() {
        patchState(store, { isAddUpdate: false })
      },
    })
  ),
  withHooks({
    onInit(store) {
      toObservable(store.loadList).subscribe((flag) => {
        if (flag) {
          store.loadNonConformanceResponseTypes();
        }
      });
    },
  })
);
