import { NgClass, NgTemplateOutlet } from '@angular/common';
import { ChangeDetectorRef, Component, inject, Input, OnInit, signal } from '@angular/core';
import { FormArray, FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { MatButtonModule } from '@angular/material/button';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { MatSelectModule } from '@angular/material/select';
import { MatCardModule } from '@angular/material/card';
import { MatCheckboxModule } from '@angular/material/checkbox';
import { ActivatedRoute, Router, RouterLink } from '@angular/router';
import { BaseComponent } from '../../../base.component';

import { ToastrService } from '@core/services/toastr-service';
import { validateFile } from '@core/domain-classes/extension-types';
import { CommonService } from '@core/services/common.service';
import { AllowFileExtension } from '@core/domain-classes/allow-file-extension';
import { catchError, concatMap, from, mergeMap, Observable, of, toArray } from 'rxjs';
import { CommonError } from '@core/error-handler/common-error';
import { DocumentChunkStatus } from '@core/domain-classes/document-chunk-status';
import { environment } from '@environments/environment';
import { NonConformanceResponseTypeStore } from '../../../non-conformance/non-conformance-response-type/store/non-conformance-response-type-store';
import { TextEditorComponent } from '@shared/text-editor/text-editor.component';
import { NonConformanceService } from '../../non-conformance.service';
import { NonConformance } from '../../model/non-conformance';
import { NonConformanceResponse } from '../../model/non-conformance-response';
import { NonConformanceResponseChunkStatus } from '../../model/non-conformance-response-chunk-status';
import { NonConformanceResponseDocument } from '../../model/non-conformance-response-document';
import { NonConformanceResponseService } from '../non-conformance-response.service';
import { OverlayPanel } from '@shared/overlay-panel/overlay-panel.service';
import { AttachmentMediaPreview } from '../../model/attachment-media-preview';
import { MatIconModule } from '@angular/material/icon';
import { MatExpansionModule } from '@angular/material/expansion';
import { UserStore } from '../../../user/store/user.store';
import { DocumentView } from '@core/domain-classes/document-view';
import { BasePreviewComponent } from '@shared/base-preview/base-preview.component';
import { QmsModuleEnum } from '@core/domain-classes/qms-module-enum';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { MatDialog } from '@angular/material/dialog';
import { ManageNonConformanceResponseTypeComponent } from '../../non-conformance-response-type/manage-non-conformance-response-type/manage-non-conformance-response-type.component';
import { LimitToPipe } from "../../../shared/pipes/limit-to.pipe";

@Component({
  selector: 'app-non-conformance-response',
  imports: [
    ReactiveFormsModule,
    MatButtonModule,
    MatSelectModule,
    MatDatepickerModule,
    TextEditorComponent,
    MatCardModule,
    MatCheckboxModule,
    NgClass,
    RouterLink,
    MatIconModule,
    MatExpansionModule,
    NgTemplateOutlet,
    PageHelpTextComponent,
    HasClaimDirective,
    TranslateModule,
    LimitToPipe
],
  templateUrl: './manage-non-conformance-response.component.html',
  styleUrl: './manage-non-conformance-response.component.scss'
})
export class NonConformanceResponseComponent extends BaseComponent implements OnInit {
  @Input() nonConformance: NonConformance;
  @Input() isDetails: boolean = false;

  readonly expandedPanelIndex = signal<number | null>(null);
  nonConformanceResponseForm: FormGroup;
  fb = inject(FormBuilder);
  route = inject(ActivatedRoute);
  ncId: string | null = null;
  nonConformanceService = inject(NonConformanceService);
  nonConformanceResponseService = inject(NonConformanceResponseService);
  cd = inject(ChangeDetectorRef);
  toastrService = inject(ToastrService);
  userStore = inject(UserStore);
  extension: string = '';
  commonService = inject(CommonService);
  allowFileExtension: AllowFileExtension[] = [];
  counter: number;
  chunkSize = environment.chunkSize;
  chunkUploads: Observable<any>[] = [];
  nonConformanceResponseTypeStore = inject(NonConformanceResponseTypeStore);
  fileArray: any[] = [];
  nonConformanceAttachmentMedia: AttachmentMediaPreview;
  overlay = inject(OverlayPanel);
  isDragging: boolean = false;
  minDate = new Date();
  commonDialogService = inject(CommonDialogService);
  router = inject(Router);
  isRoute: boolean = false;
  dialog = inject(MatDialog);

  get nonConformanceResponse() {
    return this.nonConformanceResponseForm.get('nonConformanceResponse') as FormArray;
  }

  getFilesArray(index: number): FormArray {
    const responseGroup = this.nonConformanceResponse.at(index) as FormGroup;
    return responseGroup.get('files') as FormArray;
  }

  ngOnInit(): void {
    this.getAllAllowFileExtension();
    this.creatNonConformanceResponseForm();

    // this will be used to fetch the non-conformance response details if available
    if (this.nonConformance) {
      this.isRoute = false;
      const responses = this.nonConformance.nonConformanceResponses ?? [];
      if (responses.length > 0) {
        responses.forEach(response => {
          const responseGroup = this.creatObjectNonConformanceResponse();
          responseGroup.patchValue({ ...response, isEdit: true });
          this.nonConformanceResponse.push(responseGroup);
          this.fileArray.push(
            (response.nonConformanceRespAttachments ?? []).map(file => ({
              isSuccess: true,
              name: file.fileName,
              extension: file.extension,
              totalChunk: file.totalChunk,
              id: file.id,
            }))
          );
        });
      } else {
        this.addNonConformanceResponse();
      }
    } else {
      this.isRoute = true;
      // this will be used to fetch the edit-non-conformance-response
      this.sub$.sink = this.route.data.subscribe(params => {
        const ncResponse: NonConformanceResponse = params['nonConformanceResponseDetail'];
        if (ncResponse) {
          const responseGroup = this.creatObjectNonConformanceResponse();
          responseGroup.patchValue({ ...ncResponse, isEdit: true });
          this.nonConformanceResponse.push(responseGroup);
          this.fileArray.push(
            (ncResponse.nonConformanceRespAttachments ?? []).map(file => ({
              isSuccess: true,
              name: file.fileName,
              extension: file.extension,
              totalChunk: file.totalChunk,
              id: file.id,
            }))
          );
          this.ncId = ncResponse.nonConformanceId;
          this.expandedPanelIndex.set(0);
        } else {
          this.addNonConformanceResponse();
        }
      });
    }
  }

  onDragOver(event: DragEvent): void {
    event.preventDefault();
    event.stopPropagation();
    this.isDragging = true;
  }

  onDragLeave(): void {
    this.isDragging = false;
  }

  async onDrop(event: DragEvent, index: number) {
    event.preventDefault();
    event.stopPropagation();
    this.isDragging = false;

    if (event.dataTransfer?.files) {
      await this.onFileUpload(event.dataTransfer?.files, index);
    }
  }

  creatNonConformanceResponseForm() {
    this.nonConformanceResponseForm = this.fb.group({
      nonConformanceResponse: this.fb.array([]),
    });
  }

  creatObjectNonConformanceResponse() {
    return this.fb.group({
      id: [''],
      title: ['', [Validators.required]],
      nonConformanceId: [''],
      nonConformanceResponseTypeId: ['', [Validators.required]],
      responsiblePersonId: [''],
      responseDescription: ['', [Validators.required]],
      verifiedById: [''],
      dueDate: [''],
      completionDate: [''],
      verificationDate: [''],
      isEffective: [false],
      files: this.fb.array([]),
      isEdit: [false],
    });
  }

  addNonConformanceResponse(): void {
    if (this.nonConformanceResponse.valid) {
      this.sub$.sink = this.route.paramMap.subscribe(params => {
        const ncId = params.get('ncId') || params.get('id');
        if (!ncId) {
          this.toastrService.error(this.translationService.getValue('NON_CONFORMANCE_ID_NOT_FOUND'));
          return;
        } else {
          this.nonConformanceResponse.insert(0, this.creatObjectNonConformanceResponse());
          this.fileArray.unshift([]);
          const lastIndex = this.nonConformanceResponse.length - 1;
          this.nonConformanceResponse.at(0).patchValue({ nonConformanceId: ncId });

          this.expandedPanelIndex.set(null);
          setTimeout(() => { this.expandedPanelIndex.set(0) }, 0)
        }
      });
    } else {
      this.nonConformanceResponse.markAllAsTouched();
    }
  }

  async onFileSelected(event: Event, index: number) {
    const input = event.target as HTMLInputElement;
    await this.onFileUpload(input.files, index);
    input.value = '';
  }


  async onFileUpload(files: FileList | null, index: number) {
    if (!files) return;

    for (let i = 0; i < files.length; i++) {
      if (! await validateFile(files[i])) {
        this.toastrService.error(this.translationService.getValue('INVALID_EXTENSION_OR_CORRUPT_INVALID_SIGNATURE'));
        this.cd.markForCheck();
        continue;
      }
      const file = files[i];
      this.extension = file.name.split('.').pop() ?? '';
      if (this.fileExtesionValidation(this.extension)) {
        const filesArray = this.getFilesArray(index);
        filesArray.push(
          this.fb.group({
            fileName: [file.name],
            file: [file],
            name: [file.name, [Validators.required]],
            extension: [this.extension],
            message: [''],
            isSuccess: [false],
            isLoading: [false],
            id: [''],
            totalChunk: [Math.ceil(file.size / this.chunkSize)],
          })
        );
      }
    }
    this.cd.markForCheck();
  }

  fileExtesionValidation(extension: string): boolean {
    const allowTypeExtenstion = this.allowFileExtension.find((c) =>
      c.extensions?.find((ext) => ext.toLowerCase() === extension.toLowerCase())
    );
    return allowTypeExtenstion ? true : false;
  }

  getAllAllowFileExtension() {
    this.sub$.sink = this.commonService.allowFileExtension$.subscribe(
      (allowFileExtension: AllowFileExtension[]) => {
        if (allowFileExtension) {
          this.allowFileExtension = allowFileExtension;
        }
      }
    );
  }

  removeFile(fileIndex: number, responseIndex: number): void {
    const files = this.getFilesArray(responseIndex);
    files.removeAt(fileIndex);
  }

  onSubmit(index: number) {
    const responseGroup = this.nonConformanceResponse.at(index) as FormGroup;

    if (!responseGroup.valid) {
      responseGroup.markAllAsTouched();
      return;
    }

    const saveResponse = responseGroup.getRawValue() as NonConformanceResponse;

    if (!saveResponse.isEdit) {
      this.sub$.sink = this.nonConformanceResponseService
        .creatNonConformanceResponse(saveResponse)
        .subscribe(response => {
          onSuccess(response as NonConformanceResponse);
        });
    } else {
      this.sub$.sink = this.nonConformanceResponseService
        .updateNonConformanceResponse(saveResponse)
        .subscribe(response => {
          onSuccess(response as NonConformanceResponse);
          if (this.isRoute) {
            this.router.navigate(['/nc/conformances']);
          }
        });
    }

    const onSuccess = (ncResponse: NonConformanceResponse) => {
      this.toastrService.success(
        this.translationService.getValue(
          saveResponse.isEdit
            ? 'NON_CONFORMANCE_RESPONSE_UPDATED_SUCCESSFULLY'
            : 'NON_CONFORMANCE_RESPONSE_CREATED_SUCCESSFULLY'
        )
      );

      // Update form state
      this.nonConformanceResponse.at(index).patchValue({
        isEdit: true,
        id: ncResponse.id
      });

      // Now handle document uploads
      const filesArray = this.getFilesArray(index);
      if (filesArray.length === 0) {
        return;
      }

      const concatObservable$: Observable<any>[] = [];

      filesArray.controls.forEach(control => {
        if (!control.get('isSuccess')?.value) {
          const documentObj = this.buildDocumentObject(index);
          documentObj.fileName = control.get('name')?.value;
          documentObj.extension = control.get('extension')?.value;
          documentObj.fileSize = control.get('file')?.value.size;
          documentObj.totalChunk = control.get('totalChunk')?.value;
          documentObj.nonConformanceResponseId = ncResponse.id;

          concatObservable$.push(this.saveDocumentChunk({ ...documentObj }));
        }
      });

      if (concatObservable$.length === 0) {
        return;
      }

      this.chunkUploads = [];
      this.counter = 0;

      this.sub$.sink = from(concatObservable$)
        .pipe(
          concatMap(obs =>
            obs.pipe(
              catchError(err => of(`${err.error[0] || 'Upload failed'}`))
            )
          )
        )
        .subscribe({
          next: (document: NonConformanceResponseDocument | string) => {
            this.counter++;
            if (typeof document !== 'string') {
              filesArray.at(this.counter - 1).patchValue({
                id: document.id,
                isSuccess: true
              });
              const fileData = filesArray.at(this.counter - 1).get('file')?.value;
              const totalChunk = filesArray.at(this.counter - 1).get('totalChunk')?.value;
              this.chunkUploads.push(this.uploadFile(document.id ?? '', fileData, totalChunk));
            } else {
              filesArray.at(this.counter - 1).patchValue({
                isSuccess: false
              });
            }
          },
          error: () => {
            if (this.counter < filesArray.length) {
              filesArray.at(this.counter).patchValue({
                isLoading: false,
                id: ''
              });
            }
          },
          complete: () => {
            this.uploadedChunksAllFile(index);
          }
        });
    };
  }


  buildDocumentObject(index: number): NonConformanceResponseDocument {
    return {
      description: this.nonConformanceResponseForm.getRawValue().nonConformanceResponse[index].responseDescription,
    }
  }

  uploadedChunksAllFile(index: number) {
    const resultArray: any[] = [];
    resultArray[index] = []; // initialize once!

    this.sub$.sink = from(this.chunkUploads)
      .pipe(
        concatMap((obs) =>
          obs.pipe(
            catchError(err => of(`${err.error[0] || 'Chunk upload failed'}`))
          )
        )
      )
      .subscribe({
        next: (data: NonConformanceResponseChunkStatus) => {

          if (data.status) {
            const filesArray = this.getFilesArray(index);
            const arrayValue = filesArray.getRawValue();
            const document = arrayValue.find(c => c.id === data.nonConformanceRespAttachmentId);
            if (document) {
              resultArray[index].push({
                isSuccess: true,
                id: document.id,
                name: document.name,
                extension: document.extension,
                totalChunk: document.totalChunk,
              });
            }
          } else {
            this.documentChunkRemove(data.nonConformanceRespAttachmentId);
          }
        },
        complete: () => {
          const filesArray = this.getFilesArray(index);
          filesArray.clear(); // reset form array

          const existingFiles = this.fileArray[index] ?? [];
          const newFiles = resultArray[index].map((file: any) => ({
            isSuccess: true,
            name: file.name,
            extension: file.extension,
            id: file.id,
            totalChunk: file.totalChunk,
          }));

          this.fileArray[index] = [...existingFiles, ...newFiles];
        }
      });
  }

  documentChunkRemove(documentId: string) {
    this.sub$.sink = this.nonConformanceResponseService.completeChunkDocument(documentId, false)
      .subscribe({
        next: (data) => { }
      });
  }

  uploadFile(documentId: string, file: File, totalChunk: number): Observable<any> {
    const { parallelCalls } = this.commonService.getNetworkSpeed();
    const chunkUploads: Observable<any>[] = [];
    for (let i = 0; i < totalChunk; i++) {
      const start = i * this.chunkSize;
      const end = Math.min(start + this.chunkSize, file.size);
      const chunk = file.slice(start, end);
      const formData = new FormData();
      formData.append('file', chunk);
      formData.append('chunkIndex', i.toString());
      formData.append('size', this.chunkSize.toString());
      formData.append('extension', this.extension);
      formData.append('NonConformanceRespAttachmentId', documentId);
      chunkUploads.push(this.uploadChunk(formData));
    }
    return from(chunkUploads).pipe(
      mergeMap(upload => upload, parallelCalls), // Uploads max 5 chunks at a time
      toArray(),
      concatMap(() => {
        return this.completeUpload(documentId)
      }),
      catchError(err => {
        const documentChunkStatus: DocumentChunkStatus = {
          documentId: documentId,
          status: false
        };
        return of(documentChunkStatus);
      },
      )
    );
  }

  getFileIcon(fileType: string): string {
    if (fileType.startsWith('image/')) {
      return 'image';
    } else if (fileType.includes('pdf')) {
      return 'picture_as_pdf';
    } else if (fileType.includes('spreadsheet') || fileType.includes('excel')) {
      return 'table_chart';
    } else if (fileType.includes('document') || fileType.includes('word')) {
      return 'description';
    } else {
      return 'insert_drive_file';
    }
  }

  uploadChunk(formData: FormData): Observable<NonConformanceResponseChunkStatus | CommonError> {
    return this.nonConformanceResponseService.uploadChunkDocument(formData);
  }

  completeUpload(documentId: string): Observable<any> {
    return this.nonConformanceResponseService.completeChunkDocument(documentId, true);
  }

  saveDocumentChunk(documentObj: NonConformanceResponseDocument): Observable<NonConformanceResponseDocument | CommonError> {
    return this.nonConformanceResponseService.addFileAttachment(documentObj)
  }

  onMeidaView(document: NonConformanceResponseDocument) {
    const documentView: DocumentView = {
      documentId: document.id,
      name: document.name,
      extension: document.extension,
      isVersion: false,
      isFromPublicPreview: false,
      isPreviewDownloadEnabled: true,
      isFileRequestDocument: false,
      totalChunk: document.totalChunk,
      moduleNo: QmsModuleEnum.NonConformance,
    };
    this.overlay.open(BasePreviewComponent, {
      position: 'center',
      origin: 'global',
      panelClass: ['file-preview-overlay-container', 'white-background'],
      data: documentView,
    });
  }

  deleteNonConformanceResponseFile(fileIndex: number, respIndex: number) {
    this.commonDialogService.deleteConfirmtionDialog(
      this.translationService.getValue("ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_RESPONSE_ATTACHMENT"),
    ).subscribe({
      next: (result: boolean) => {
        if (result) {
          const files = this.fileArray[respIndex];
          const fileId = files[fileIndex].id;

          if (fileId) {
            this.nonConformanceResponseService.deleteResponseAttachment(fileId).subscribe(() => {
              files.splice(fileIndex, 1);
              this.toastrService.success(this.translationService.getValue("RESPONSE_ATTACHMENT_DELETED_SUCCESSFULLY"));
            });
          }
        }
      }
    });
  }

  addNonConformanceResponseType(): void {
    const dialogRef = this.dialog.open(ManageNonConformanceResponseTypeComponent, {
      width: '600px',
    });
    dialogRef.afterClosed().subscribe((result) => {
      if (result) {
        const expandedIndex = this.expandedPanelIndex();
        if (expandedIndex !== null) {
          this.nonConformanceResponse.at(expandedIndex).get('nonConformanceResponseTypeId')?.setValue(result.id);
        }
      }
    });
  }
}
