import { Component, Input, OnChanges, SimpleChanges } from '@angular/core';
import { MatTableModule } from '@angular/material/table';
import { inject } from '@angular/core';
import { NonConformanceResponse } from '../../model/non-conformance-response';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { TranslationService } from '@core/services/translation.service';

import { ToastrService } from '@core/services/toastr-service';
import { RouterModule } from '@angular/router';
import { NonConformance } from '../../model/non-conformance';
import { NonConformanceService } from '../../non-conformance.service';
import { MatCheckboxModule } from '@angular/material/checkbox';
import { User } from '@core/domain-classes/user';
import { NonConformanceResponseService } from '../../non-conformance-responses/non-conformance-response.service';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { MatButtonModule } from '@angular/material/button';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';
import { NgClass } from '@angular/common';

@Component({
  selector: 'app-non-conformance-response-list',
  standalone: true,
  imports: [
    MatTableModule,
    RouterModule,
    MatCheckboxModule,
    PageHelpTextComponent,
    HasClaimDirective,
    TranslateModule,
    MatButtonModule,
    UTCToLocalTime,
    MatIconModule,
    MatCardModule,
    NgClass
  ],
  templateUrl: './non-conformance-response-list.component.html',
  styleUrl: './non-conformance-response-list.component.scss'
})
export class NonConformanceResponseListComponent implements OnChanges {
  @Input() nonConformance!: NonConformance;
  @Input() userList!: User[];
  data: NonConformanceResponse[] = [];
  subActionColumnToDisplay = ['action', 'ncTitle', 'nonConformanceResponseType', 'responsiblePerson', 'verifiedBy', 'dueDate', 'completedDate', 'verifiedDate', 'isEffective'];
  commandDialogService = inject(CommonDialogService);
  translationService = inject(TranslationService);
  toastrService = inject(ToastrService);
  nonConformaceService = inject(NonConformanceService);
  nonConformaceResponseService = inject(NonConformanceResponseService);

  ngOnChanges(changes: SimpleChanges): void {
    if (changes['nonConformance']) {
      this.getNonConformanceResponse();
    }
  }

  getNonConformanceResponse() {
    if (this.nonConformance && this.nonConformance.id) {
      this.nonConformaceService.getNonConformanceById(this.nonConformance.id).subscribe({
        next: (response: NonConformance) => {
          this.data = response.nonConformanceResponses as NonConformanceResponse[];
        },
        error: () => {
          this.toastrService.error(this.translationService.getValue("FAILED_TO_LOAD_RESPONSES"));
        }
      });
    } else {
      this.data = [];
    }
  }

  deleteNonConformanceResponse(responseId: string) {
    this.commandDialogService.deleteConfirmtionDialog(
      this.translationService.getValue("ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_RESPONSE")
    ).subscribe({
      next: (result: boolean) => {
        if (result) {
          this.nonConformaceResponseService.deleteNonConformanceResponse(responseId).subscribe(() => {
            this.data = this.data.filter(response => response.id !== responseId);
            this.toastrService.success(this.translationService.getValue("RESPONSE_DELETED_SUCCESSFULLY"));
          });
        }
      }
    });
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.data.indexOf(row);
  }
}
