import { HttpClient, HttpParams, HttpResponse } from '@angular/common/http';
import { inject, Injectable } from '@angular/core';
import { NonConformance } from '../non-conformance/model/non-conformance';
import { CommonError } from '@core/error-handler/common-error';
import { CommonHttpErrorService } from '@core/error-handler/common-http-error.service';;
import { catchError, Observable } from 'rxjs';
import { NonConformanceResource } from './model/non-conformance-resource';
import { IdName } from '@core/domain-classes/id-name';

@Injectable({
  providedIn: 'root'
})

export class NonConformanceService {
  private httpClient = inject(HttpClient)
  private commonHttpErrorService = inject(CommonHttpErrorService)
  constructor() { }

  getNonConformance(
    resource: NonConformanceResource
  ): Observable<HttpResponse<NonConformance[]>> {
    const url = 'NonConformance';
    const customParams = new HttpParams()
      .set('OrderBy', resource.orderBy)
      .set('PageSize', resource.pageSize)
      .set('Skip', resource.skip)
      .set('NCNumber', resource.ncNumber)
      .set('ReportedById', resource.reportedById)
      .set('Severity', resource.severity)
      .set('Status', resource.status)
      .set('DepartmentId', resource.departmentId)
      .set('AuditId', resource.auditId)
      .set('Title', resource.title)
    return this.httpClient
      .get<NonConformance[]>(url, {
        params: customParams,
        observe: 'response',
      });

  }

  addNonConformance(nonConformance: NonConformance): Observable<NonConformance | CommonError> {
    const url = 'NonConformance';
    return this.httpClient
      .post<NonConformance>(url, nonConformance)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  updateNonConformance(nonConformance: NonConformance): Observable<NonConformance | CommonError> {
    const url = `NonConformance/${nonConformance.id}`;
    return this.httpClient
      .put<NonConformance>(url, nonConformance)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  deleteNonConformance(nonConformanceId: string): Observable<void | CommonError> {
    const url = `NonConformance/${nonConformanceId}`;
    return this.httpClient
      .delete<void>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getNonConformanceById(nonConformanceId: string): Observable<NonConformance> {
    const url = `NonConformance/${nonConformanceId}`;
    return this.httpClient
      .get<NonConformance>(url);

  }

  getNonConformanceNumber(): Observable<NonConformance | CommonError> {
    const url = 'NonConformance/GetNewNcNumber';
    return this.httpClient
      .get<NonConformance>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getAllAudits(): Observable<IdName[] | CommonError> {
    const url = 'NonConformance/audits';
    return this.httpClient.get<IdName[]>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

}
