import { patchState, signalStore, withHooks, withMethods, withState } from "@ngrx/signals";
import { rxMethod } from "@ngrx/signals/rxjs-interop";
import { NonConformanceService } from "../non-conformance.service";
import { inject } from "@angular/core";
import { ToastrService } from '@core/services/toastr-service';
import { TranslationService } from "@core/services/translation.service";
import { distinctUntilChanged, pipe, switchMap, tap } from "rxjs";
import { tapResponse } from "@ngrx/operators";
import { NonConformance } from "../../non-conformance/model/non-conformance";
import { toObservable } from "@angular/core/rxjs-interop";
import { NonConformanceResource } from "../model/non-conformance-resource";
import { HttpResponse } from "@angular/common/http";

type NonConformanceState = {
    nonConformanceList: NonConformance[];
    nonConformance: NonConformance;
    loadList: boolean;
    isAddUpdate: boolean;
    filterParameters: NonConformanceResource;
};

export const initialNonConformanceState: NonConformanceState = {
    nonConformanceList: [],
    nonConformance: {} as NonConformance,
    loadList: true,
    isAddUpdate: false,
    filterParameters: {
        ncNumber: '',
        reportedById: '',
        severity: '',
        status: '',
        departmentId: '',
        auditId: '',
        orderBy: 'ncNumber asc',
        pageSize: 10,
        skip: 0,
        totalCount: 0,
        title: ''
    }
};

export const NonConformanceStore = signalStore(
    { providedIn: 'root' },
    withState(initialNonConformanceState),
    withMethods(
        (
            store,
            nonConformanceService = inject(NonConformanceService),
            toastrService = inject(ToastrService),
            translationService = inject(TranslationService),
        ) => ({
            loadByQuery: rxMethod<NonConformanceResource>(
                pipe(
                    tap(() => patchState(store, { nonConformanceList: [] })),
                    switchMap((filter) => {
                        return nonConformanceService.getNonConformance(filter).pipe(
                            tapResponse({
                                next: (httpResponse: HttpResponse<NonConformance[]>) => {
                                    if (httpResponse && httpResponse.body) {
                                        patchState(store, { nonConformanceList: [...httpResponse.body], loadList: false })
                                    }
                                    if (httpResponse.headers.get('x-pagination')) {
                                        const pagination: NonConformanceResource = JSON.parse(httpResponse.headers.get('x-pagination') ?? '');
                                        patchState(store, { filterParameters: { ...filter, totalCount: pagination.totalCount } })
                                    }
                                },
                                error: (error: any) => {
                                    toastrService.error(error.error);
                                }
                            })
                        );
                    })
                )
            ),
            addNonConformance: rxMethod<NonConformance>(
                pipe(
                    distinctUntilChanged(),
                    tap(() => patchState(store, { loadList: false })),
                    switchMap((nonConformance: NonConformance) =>
                        nonConformanceService.addNonConformance(nonConformance).pipe(
                            tapResponse({
                                next: () => {
                                    toastrService.success(
                                        translationService.getValue('NON_CONFORMANCE_CREATED_SUCCESSFULLY')
                                    );
                                    patchState(store, { isAddUpdate: true, loadList: true });
                                },
                                error: (err: any) => {
                                    console.error(err);
                                },
                            })
                        )
                    )
                )
            ),
            updateNonConformance: rxMethod<NonConformance>(
                pipe(
                    distinctUntilChanged(),
                    tap(() => patchState(store, { loadList: false })),
                    switchMap((nonConformance: NonConformance) =>
                        nonConformanceService.updateNonConformance(nonConformance).pipe(
                            tapResponse({
                                next: (response) => {
                                    const updatedNonConformance: NonConformance = response as NonConformance;
                                    toastrService.success(
                                        translationService.getValue('NON_CONFORMANCE_UPDATED_SUCCESSFULLY')
                                    );
                                    patchState(store, {
                                        nonConformanceList: store.nonConformanceList().map((nc) => {
                                            return updatedNonConformance.id === nc.id ? updatedNonConformance : nc;
                                        }),
                                        isAddUpdate: true,
                                    });
                                },
                                error: (err: any) => {
                                    console.error(err);
                                },
                            })
                        )
                    )
                )
            ),
            deleteNonConformanceById: rxMethod<string>(
                pipe(
                    distinctUntilChanged(),
                    tap(() => patchState(store, { loadList: false })),
                    switchMap((nonConformanceId: string) =>
                        nonConformanceService.deleteNonConformance(nonConformanceId).pipe(
                            tapResponse({
                                next: () => {
                                    toastrService.success(
                                        translationService.getValue('NON_CONFORMANCE_DELETED_SUCCESSFULLY')
                                    );
                                    patchState(store, {
                                        nonConformanceList: store.nonConformanceList().filter((w) => w.id !== nonConformanceId),
                                        filterParameters: {
                                            ...store.filterParameters(),
                                            totalCount: store.filterParameters().totalCount - 1,
                                        },
                                    });
                                },
                                error: (err: any) => {
                                    console.error(err);
                                },
                            })
                        )
                    )
                )
            ),
            resetflag() {
                patchState(store, { isAddUpdate: false })
            },
        })
    ),
    withHooks({
        onInit(store) {
            toObservable(store.loadList).subscribe((flag) => {
                if (flag) {
                    store.loadByQuery(store.filterParameters());
                }
            });
        },
    })
);
