import {
  AfterViewInit,
  Component,
  ElementRef,
  OnInit,
  ViewChild,
} from '@angular/core';
import { MatPaginator } from '@angular/material/paginator';
import { MatSort } from '@angular/material/sort';
import { ResponseHeader } from '@core/domain-classes/document-header';
import { DocumentResource } from '@core/domain-classes/document-resource';
import { NotificationType, UserNotification } from '@core/domain-classes/notification';
import { OverlayPanel } from '@shared/overlay-panel/overlay-panel.service';
import { fromEvent, merge, Observable } from 'rxjs';
import { debounceTime, distinctUntilChanged, tap } from 'rxjs/operators';
import { BaseComponent } from '../../base.component';
import { NotificationDataSource } from '../notification-datassource';
import { NotificationService } from '../notification.service';
import { DocumentView } from '@core/domain-classes/document-view';
import { ToastrService } from '@core/services/toastr-service';
import { ServiceResponse } from '../../core/domain-classes/service-response';
import { WorkflowInstanceService } from '../../workflows/workflow-instance.service';
import { VisualWorkflowInstance } from '@core/domain-classes/visual-workflow-instance';
import { MatDialog } from '@angular/material/dialog';
import { DocumentService } from '../../document/document.service';
import { DocumentInfo } from '@core/domain-classes/document-info';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { MatTableModule } from '@angular/material/table';
import { TranslateModule } from '@ngx-translate/core';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { MatCardModule } from '@angular/material/card';
import { MatIconModule } from '@angular/material/icon';
import { MatButtonModule } from '@angular/material/button';
import { Router } from '@angular/router';

@Component({
  selector: 'app-notification-list',
  templateUrl: './notification-list.component.html',
  styleUrls: ['./notification-list.component.scss'],
  standalone: true,
  imports: [
    PageHelpTextComponent,
    MatPaginator,
    MatSort,
    MatTableModule,
    TranslateModule,
    UTCToLocalTime,
    MatCardModule,
    MatIconModule,
    MatButtonModule
  ],
})
export class NotificationListComponent
  extends BaseComponent
  implements OnInit, AfterViewInit {
  dataSource: NotificationDataSource;
  notifications: UserNotification[] = [];
  displayedColumns: string[] = ['action', 'createdDate', 'message'];
  notificationResource: DocumentResource;
  @ViewChild(MatPaginator) paginator: MatPaginator;
  @ViewChild(MatSort) sort: MatSort;
  @ViewChild('input') input: ElementRef;
  footerToDisplayed = ['footer'];
  constructor(
    private notificationService: NotificationService,
    private toastrService: ToastrService,
    public overlay: OverlayPanel,
    private workflowInstanceService: WorkflowInstanceService,
    private dialog: MatDialog,
    private documentService: DocumentService,
    private router: Router
  ) {
    super();
    this.notificationResource = new DocumentResource();
    this.notificationResource.pageSize = 10;
    this.notificationResource.orderBy = 'createdDate desc';
  }

  ngOnInit(): void {
    this.dataSource = new NotificationDataSource(this.notificationService);
    this.dataSource.loadNotifications(this.notificationResource);
    this.getResourceParameter();
  }

  ngAfterViewInit() {
    this.sort.sortChange.subscribe(() => (this.paginator.pageIndex = 0));

    this.sub$.sink = merge(this.sort.sortChange, this.paginator.page)
      .pipe(
        tap(() => {
          this.notificationResource.skip = this.paginator.pageIndex * this.paginator.pageSize;
          this.notificationResource.pageSize = this.paginator.pageSize;
          this.notificationResource.orderBy =
            this.sort.active + ' ' + this.sort.direction;
          this.dataSource.loadNotifications(this.notificationResource);
        })
      )
      .subscribe();

    this.sub$.sink = fromEvent(this.input.nativeElement, 'keyup')
      .pipe(
        debounceTime(1000),
        distinctUntilChanged(),
        tap(() => {
          this.paginator.pageIndex = 0;
          this.notificationResource.skip = 0;
          this.notificationResource.name = this.input.nativeElement.value;
          this.dataSource.loadNotifications(this.notificationResource);
        })
      )
      .subscribe();
  }
  getResourceParameter() {
    this.sub$.sink = this.dataSource.responseHeaderSubject$.subscribe(
      (c: ResponseHeader) => {
        if (c) {
          this.notificationResource.pageSize = c.pageSize;
          this.notificationResource.skip = c.skip;
          this.notificationResource.totalCount = c.totalCount;
        }
      }
    );
  }

  viewDocument(notification: UserNotification) {
    if (notification.notificationsType === NotificationType.REMINDER) {
      this.notificationService
        .checkReminderByDocumentId(notification.documentId ?? '')
        .subscribe((response: ServiceResponse<boolean>) => {
          if (response.data) {
            this.previewDocument(notification);
          } else {
            this.toastrService.error(
              this.translationService.getValue('YOU_ARE_NOT_AUTHORIZED_TO_VIEW_THIS_REMINDER')
            );
          }
        });
    } else if (notification.notificationsType === NotificationType.SHARE_USER) {
      this.notificationService
        .checkShareUserByDocumentId(notification.documentId ?? '')
        .subscribe((isShare: ServiceResponse<boolean>) => {
          if (isShare.data) {
            this.previewDocument(notification);
          } else {
            this.toastrService.error(
              this.translationService.getValue('YOU_ARE_NOT_AUTHORIZED_TO_VIEW_THIS_DOCUMENT')
            );
          }
        });
    } else if (notification.notificationsType === NotificationType.WORKFLOW) {
      this.previewDocument(notification);
    }
  }

  async previewDocument(notification: UserNotification) {
    this.documentService.getDocument(notification.documentId ?? '')
      .subscribe(async (document: DocumentInfo) => {
        try {
          const urls = document.url?.split('.') ?? [];
          const extension = urls[1];
          const documentView: DocumentView = {
            documentId: document.id,
            name: document.name,
            url: document.url,
            extension: extension,
            isVersion: false,
            isFromPublicPreview: false,
            isPreviewDownloadEnabled: false,
            isFileRequestDocument: false,
            isSignatureExists: document.isSignatureExists,
            documentNumber: document.documentNumber
          };
          const { BasePreviewComponent } = await import(
            '../../shared/base-preview/base-preview.component'
          );
          this.overlay.open(BasePreviewComponent, {
            position: 'center',
            origin: 'global',
            panelClass: ['file-preview-overlay-container', 'white-background'],
            data: documentView,
          });
        }
        finally { }
      });
  }
  viewVisualWorkflow(notification: UserNotification): void {
    this.workflowInstanceService
      .getvisualWorkflowInstance(notification.workflowInstanceId ?? '')
      .subscribe({
        next: async (data: VisualWorkflowInstance) => {
          try {
            const { VisualWorkflowGraphComponent } = await import(
              '../../workflows/visual-workflow-graph/visual-workflow-graph.component'
            );
            const dialogRef = this.dialog.open(VisualWorkflowGraphComponent, {
              width: '100%',
              maxWidth: '70vw',
              data: Object.assign({}, data),
            });
          }
          finally { }
        },
        error: (error) => { },
      });
  }

  getNotificationMessage(notification: UserNotification): string {
    if (notification.jsonPayload) {
      const payload = JSON.parse(notification.jsonPayload);

      switch (notification.notificationsType) {
        case 5: // COURSE
          return this.translationService.getValue('YOU_HAVE_BEEN_ASSIGNED_THE_COURSE') + `"${payload.Title}"` + this.translationService.getValue('PLEASE_COMPLETE_IT_AS_SCHEDULED');

        case 6: // AUDIT
          return this.translationService.getValue('YOU_HAVE_BEEN_ASSIGNED_THE_AUDIT') + `"${payload.Title}".`;

        case 7: // CAPA_ACTION
          return this.translationService.getValue('YOU_HAVE_BEEN_ASSIGNED_A_CAPA_ACTION_FOR') + `"${payload.Title}".`;

        case 8: // COMPLAINT_ACTION
          return this.translationService.getValue('YOU_HAVE_BEEN_DESIGNATED_AS_THE_RESPONSIBLE_PERSON_FOR_THE_COMPLAINT_ACTION') + `"${payload.Title}".`;

        case 9: // COMPLAINT_INVESTIGATION
          return this.translationService.getValue('YOU_HAVE_BEEN_ASSIGNED_TO_INVESTIGATE_THE_COMPLAINT') + `"${payload.Title}".`;

        case 10: // NC_RESPONSE
          return this.translationService.getValue('YOU_HAVE_BEEN_DESIGNATED_AS_THE_RESPONSIBLE_PERSON_FOR_THE_NON_CONFORMANCE') + `"${payload.Title}".`;

        case 11: // RISK
          return this.translationService.getValue('YOU_HAVE_BEEN_ASSIGNED_TO_REVIEW_HANDLE_THE_RISK') + `"${payload.Title}".`;

        case 12: // RISK_CONTROL
          return this.translationService.getValue('YOU_HAVE_BEEN_DESIGNATED_AS_THE_RESPONSIBLE_PERSON_FOR_THE_RISK_CONTROL_TASK') + `"${payload.Title}".`;

        case 13: // RISK_REVIEW
          return this.translationService.getValue('YOU_HAVE_BEEN_DESIGNATED_AS_THE_REVIEWER_FOR_A_RISK') + `"${payload.Title}".` || payload.Message;

        case 14: // COMPLAINT
          return this.translationService.getValue('A_NEW_COMPLAINT') + `"${payload.Title}"` + this.translationService.getValue('HAS_BEEN_ASSIGNED_TO_YOU');

        case 15: // AUDITREWORK
          return this.translationService.getValue('THE_AUDIT') + `"${payload.Title}"` + this.translationService.getValue('HAS_BEEN_SENT_BACK_FOR_REWORK');

        case 16: // AUDITREJECTED
          return this.translationService.getValue('THE_AUDIT') + `"${payload.Title}"` + this.translationService.getValue('HAS_BEEN_REJECTED');

        case 17: // AUDITINREVIEW
          return this.translationService.getValue('THE_AUDIT') + `"${payload.Title}"` + this.translationService.getValue('HAS_BEEN_SUBMITTED_FOR_YOUR_REVIEW');

        default:
          return notification.message || notification.documentName || this.translationService.getValue('YOU_HAVE_A_NEW_NOTIFICATION');
      }
    } else {
      switch (notification.notificationsType) {
        case 0: // REMINDER
          return notification.message ? this.translationService.getValue('REMINDER') + ` :: "${notification.message}"` : this.translationService.getValue('YOU_HAVE_A_REMINDER_NOTIFICATION');

        case 1: // SHARE_USER
          return notification.message || notification.documentName || this.translationService.getValue('YOU_HAVE_BEEN_GRANTED_ACCESS_TO_A_FILE_OR_FOLDER');

        case 2: // WORKFLOW
          return notification.message || notification.documentName || this.translationService.getValue('A_WORKFLOW_TASK_HAS_BEEN_ASSIGNED_TO_YOU');

        case 3: // FILE_REQUEST
          return notification.message || notification.documentName || this.translationService.getValue('SOMEONE_HAS_REQUESTED_YOU_TO_UPLOAD_REQUIRED_FILES');

        case 4: // SHARE_FOLDER
          return notification.message || notification.documentName || this.translationService.getValue('A_FOLDER_HAS_BEEN_SHARED_WITH_YOU');
        default:
          return notification.message || notification.documentName || this.translationService.getValue('YOU_HAVE_A_NEW_NOTIFICATION');
      }
    }
  }

  viewNotification(notification: UserNotification) {
    switch (notification.notificationsType) {
      case 0: // REMINDER
        this.router.navigate(['/reminders']);
        break;

      case 1: // SHARE_USER
        if (notification.documentId) {
          this.router.navigate(['/assign/list-view']);
        } else {
          this.router.navigate(['/assign/folder-view']);
        }
        break;

      case 2: // WORKFLOW
        this.router.navigate(['/current-workflow']);
        break;

      case 3: // FILE_REQUEST
        this.router.navigate(['/file-request']);
        break;

      case 4: // SHARE_FOLDER
        this.router.navigate(['/assign/folder-view']);
        break;

      case 5: // COURSE
        this.router.navigate(['/course/my-courses']);
        break;

      case 6: // AUDIT
        this.router.navigate(['/audit/audit-list']);
        break;

      case 7: // CAPA_ACTION
        this.router.navigate(['/capa/actions']);
        break;

      case 8: // COMPLAINT_ACTION
        this.router.navigate(['/complaint/actions']);
        break;

      case 9: // COMPLAINT_INVESTIGATION
        this.router.navigate(['/complaint/actions']);
        break;

      case 10: // NC_RESPONSE
        this.router.navigate(['/nc/conformances']);
        break;

      case 11: // RISK
        this.router.navigate(['/risk/list']);
        break;

      case 12: // RISK_CONTROL
        this.router.navigate(['/risk/my-risk-list']);
        break;

      case 13: // RISK_REVIEW
        this.router.navigate(['/risk/my-risk-list']);
        break;

      case 14: // COMPLAINT
        this.router.navigate(['/complaint/list']);
        break;

      case 15: // AUDITREWORK
        this.router.navigate(['/audit/audit-list']);
        break;

      case 16: // AUDITREJECTED
        this.router.navigate(['/audit/audit-list']);
        break;

      case 17: // AUDITINREVIEW
        this.router.navigate(['/audit/audit-for-review']);
        break;

      default:
        if (notification.documentId) {
          this.router.navigate(['/assign/list-view']);
        } else {
          this.router.navigate(['/']);
        }
        break;
    }
  }
}
