import { ChangeDetectorRef, Component, EventEmitter, inject, Input, OnChanges, OnInit, Output, SimpleChanges } from '@angular/core';
import { FormArray, FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { MatCardModule } from '@angular/material/card';
import { MatIconModule } from '@angular/material/icon';
import { RiskAttachment } from '../../model/risk-attachment';
import { DocumentView } from '@core/domain-classes/document-view';
import { BasePreviewComponent } from '@shared/base-preview/base-preview.component';
import { catchError, concatMap, from, mergeMap, Observable, of, toArray } from 'rxjs';

import { ToastrService } from '@core/services/toastr-service';
import { CommonService } from '@core/services/common.service';
import { ActivatedRoute } from '@angular/router';
import { RiskAttachmentService } from '../risk-attachment.service';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { environment } from '@environments/environment';
import { AllowFileExtension } from '@core/domain-classes/allow-file-extension';
import { OverlayPanel } from '@shared/overlay-panel/overlay-panel.service';
import { BaseComponent } from '../../../base.component';
import { validateFile } from '@core/domain-classes/extension-types';
import { RiskAttachmentStatus } from '../../model/risk-attachment-status';
import { CommonError } from '@core/error-handler/common-error';
import { RiskAttachmentChunk } from '../../model/risk-attachment-chunk';
import { MatButtonModule } from '@angular/material/button';
import { QmsModuleEnum } from '@core/domain-classes/qms-module-enum';
import { DragDropModule } from '@angular/cdk/drag-drop';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { NgTemplateOutlet } from '@angular/common';

@Component({
  selector: 'app-manage-risk-attachment',
  imports: [
    MatCardModule,
    ReactiveFormsModule,
    MatIconModule,
    MatButtonModule,
    DragDropModule,
    HasClaimDirective,
    TranslateModule,
    NgTemplateOutlet
  ],
  templateUrl: './manage-risk-attachment.component.html',
  styleUrl: './manage-risk-attachment.component.scss'
})
export class ManageRiskAttachmentComponent extends BaseComponent implements OnInit, OnChanges {
  @Input() riskAttachments: RiskAttachment[] = [];
  @Output() files = new EventEmitter<any>();
  fileUploadGroup!: FormGroup;
  fb = inject(FormBuilder);
  toastrService = inject(ToastrService);
  cd = inject(ChangeDetectorRef);
  commonService = inject(CommonService);
  route = inject(ActivatedRoute);
  riskAttachmentService = inject(RiskAttachmentService);
  commandDialogService = inject(CommonDialogService);
  extension: string | null = null;
  loading = false;
  counter = 0;
  chunkSize = environment.chunkSize;
  chunkUploads: Observable<any>[] = [];
  fileArray: any[] = [];
  isDragging = false;
  allowFileExtension: AllowFileExtension[] = [];
  overlay = inject(OverlayPanel);
  isAssignedRisk: boolean = false;

  get fileInputs(): FormArray {
    return this.fileUploadGroup.get('files') as FormArray;
  }

  ngOnInit(): void {
    this.createFileUploadForm();
    this.getAllAllowFileExtension()

    this.route.parent?.paramMap.subscribe((params) => {
      const riskId = params.get('id');
      if (riskId) {
        this.fileUploadGroup.patchValue({ riskId: riskId });
        this.getAllRiskAttachmentsByRequestId(riskId);
      }
    });
  }

  ngOnChanges(changes: SimpleChanges): void {
    if (changes['riskAttachments'] && this.riskAttachments) {
      this.fileArray = this.riskAttachments.map((file) => ({
        ...file,
        name: file.fileName ? file.fileName : file.name,
      }));
      this.isAssignedRisk = true;
    } else {
      this.isAssignedRisk = false;
      this.fileArray = [];
    }
  }

  createFileUploadForm(): void {
    const riskIdValue = this.riskAttachments?.[0]?.riskId || '';

    this.fileUploadGroup = this.fb.group({
      riskId: [riskIdValue, Validators.required],
      files: this.fb.array([]),
    });
  }

  getAllRiskAttachmentsByRequestId(requestId: string): void {
    this.sub$.sink = this.riskAttachmentService
      .getAllRiskAttachmentsByRiskId(requestId)
      .subscribe({
        next: (data: RiskAttachment[]) => {
          const attachments = data as RiskAttachment[];
          if (attachments) {
            this.fileArray = data.map((file) => ({
              isSuccess: true,
              name: file.fileName,
              extension: file.extension,
              id: file.id,
              totalChunk: file.totalChunk,
            }));
          }
        },
        error: (error) => {
          this.toastrService.error(error)
        }
      });
  }

  getAllAllowFileExtension() {
    this.sub$.sink = this.commonService.allowFileExtension$.subscribe(
      (allowFileExtension: AllowFileExtension[]) => {
        if (allowFileExtension) {
          this.allowFileExtension = allowFileExtension;
        }
      }
    );
  }

  onDragOver(event: DragEvent): void {
    event.preventDefault();
    event.stopPropagation();
    this.isDragging = true;
  }

  onDragLeave(): void {
    this.isDragging = false;
  }

  async onDrop(event: DragEvent) {
    event.preventDefault();
    event.stopPropagation();
    this.isDragging = false;

    if (event.dataTransfer?.files) {
      await this.fileUpload(event.dataTransfer?.files);
    }
  }

  get allowedFileTypes(): string {
    const allowedExtensionscombine = this.allowFileExtension.map(
      (ext) => ext.extension
    );
    return allowedExtensionscombine.length > 0
      ? allowedExtensionscombine.join(',')
      : '';
  }

  async onFileSelected(event: Event) {
    const input = event.target as HTMLInputElement;
    await this.fileUpload(input.files);
    input.value = '';
  }

  fileExtesionValidation(extension: string): boolean {
    const allowTypeExtenstion = this.allowFileExtension.find((c) =>
      c.extensions?.find((ext) => ext.toLowerCase() === extension.toLowerCase())
    );
    return allowTypeExtenstion ? true : false;
  }

  async fileUpload(files: FileList | null): Promise<void> {
    if (!files) return;

    for (let i = 0; i < files.length; i++) {
      if (!(await validateFile(files[i]))) {
        this.toastrService.error(
          this.translationService.getValue(
            'INVALID_EXTENSION_OR_CORRUPT_INVALID_SIGNATURE'
          )
        );
        this.cd.markForCheck();
        continue;
      }
      const file = files[i];
      this.extension = file.name?.split('.').pop() ?? '';
      if (this.fileExtesionValidation(this.extension)) {
        this.fileInputs.push(
          this.fb.group({
            fileName: [file.name],
            file: [file],
            name: [file.name, [Validators.required]],
            extension: [this.extension],
            message: [''],
            isSuccess: [false],
            isLoading: [false],
            riskId: [this.fileUploadGroup.get('riskId')?.value],
            riskAttachmentId: [''],
            totalChunk: [Math.ceil(file.size / this.chunkSize)],
          })
        );
      }
    }
  }

  removeLocalFile(index: number): void {
    if (index >= 0 && index < this.fileInputs.length) {
      this.fileInputs.removeAt(index);
    }
  }

  saveFilesAndDocument(): void {
    if (!this.fileInputs.valid) {
      this.fileInputs.markAllAsTouched();
      return;
    }

    const fileInputsArray = this.fileInputs.controls;
    if (fileInputsArray.length === 0) {
      this.loading = false;
      return;
    }

    const concatObservable$: Observable<any>[] = [];

    fileInputsArray.forEach((control) => {
      if (!control.get('isSuccess')?.value) {
        const riskRequestAttachment: RiskAttachment = {
          fileName: control.get('fileName')?.value,
          fileSize: this.formatFileSize(control.get('file')?.value.size),
          extension: control.get('extension')?.value,
          riskId: control.get('riskId')?.value,
          totalChunk: control.get('totalChunk')?.value,
        };
        concatObservable$.push(
          this.saveRiskRequestAttachmentChunk({ ...riskRequestAttachment })
        );
      }
    });

    if (concatObservable$.length === 0) {
      this.loading = false;
      return;
    }

    this.chunkUploads = [];
    this.counter = 0;

    this.sub$.sink = from(concatObservable$)
      .pipe(
        concatMap((obs) =>
          obs.pipe(
            catchError((err) => of(`${err.error?.[0] || 'Upload failed'}`))
          )
        )
      )
      .subscribe({
        next: (document: RiskAttachment) => {
          this.counter++;
          if (typeof document !== 'string') {
            this.fileInputs.at(this.counter - 1).patchValue({
              isLoading: false,
              riskAttachmentId: document.id,
              isSuccess: true,
            });
            const fileData = this.fileInputs.at(this.counter - 1).get('file')?.value;
            const totalChunk = this.fileInputs.at(this.counter - 1).get('totalChunk')?.value;
            this.chunkUploads.push(this.uploadFile(document.id ?? '', fileData, totalChunk));
          } else {
            this.fileInputs.at(this.counter - 1).patchValue({
              isLoading: false,
              isSuccess: false,
            });
          }
        },
        error: () => {
          if (this.counter < this.fileInputs.length) {
            this.fileInputs.at(this.counter).patchValue({
              isLoading: false,
              riskAttachmentId: '',
            });
          }
          this.loading = false;
        },
        complete: () => {
          if (this.chunkUploads.length > 0) {
            this.uploadedChunksAllFile();
            this.loading = false;
          }
        },
      });
  }

  uploadFile(riskAttachmentId: string, file: File, totalChunk: number): Observable<any> {
    const { parallelCalls } = this.commonService.getNetworkSpeed();
    const chunkUploads: Observable<any>[] = [];

    for (let i = 0; i < totalChunk; i++) {
      const start = i * this.chunkSize;
      const end = Math.min(start + this.chunkSize, file.size);
      const chunk = file.slice(start, end);
      const formData = new FormData();
      formData.append('file', chunk);
      formData.append('chunkIndex', i.toString());
      formData.append('size', this.chunkSize.toString());
      formData.append('extension', this.extension ?? '');
      formData.append('riskAttachmentId', riskAttachmentId);
      chunkUploads.push(this.uploadChunk(formData));
    }
    return from(chunkUploads).pipe(
      mergeMap((upload) => upload, parallelCalls), // Uploads max 5 chunks at a time
      toArray(),
      concatMap(() => {

        return this.completeUpload(riskAttachmentId, true);
      }),
      catchError((err) => {
        const documentChunkStatus: RiskAttachmentStatus = {
          riskAttachmentId: riskAttachmentId,
          status: false,
        };
        return of(documentChunkStatus);
      })
    );
  }

  uploadedChunksAllFile(): void {
    const resultArray: any[] = []
    this.sub$.sink = from(this.chunkUploads)
      .pipe(
        concatMap((obs) =>
          obs.pipe(
            catchError((err) => of(`${err.error[0] || 'Chunk upload failed'}`))
          )
        )
      )
      .subscribe({
        next: (data: RiskAttachmentStatus) => {
          this.loading = false;
          if (data.status) {
            const fileIndex = this.fileInputs.controls.findIndex(
              (ctrl) =>
                ctrl.get('riskAttachmentId')?.value ===
                data.riskAttachmentId
            );
            if (fileIndex !== -1) {
              const control = this.fileInputs.at(fileIndex);
              resultArray.push({
                isSuccess: true,
                documentId: data.riskAttachmentId,
                name: control.get('name')?.value,
                totalChunk: control.get('totalChunk')?.value,
              });
              this.toastrService.success(this.translationService.getValue('FILE_UPLOADED_SUCCESSFULLY'));
            }
          } else {
            this.documentChunkRemove(data.riskAttachmentId);
          }
        },
        error: () => {
          this.loading = false;
        },
        complete: () => {
          this.fileInputs.clear();
          const newFiles = resultArray
            .filter((f) => f.isSuccess)
            .map((file) => ({
              isSuccess: true,
              name: file.name,
              extension: file.name.split('.').pop(),
              id: file.documentId,
              totalChunk: file.totalChunk,
            }));
          this.fileArray = [...this.fileArray, ...newFiles];
          if (this.isAssignedRisk) {
            this.files.emit(newFiles);
          }
          this.cd.detectChanges();
        },
      });
  }

  formatFileSize(bytes: number): number {
    return parseInt((bytes / 1024).toFixed(1));
  }

  documentChunkRemove(documentId: string) {
    this.loading = true;
    this.sub$.sink = this.riskAttachmentService
      .markAsRiskAttachmentChunkComplete(documentId, false)
      .subscribe({
        next: (data) => {
          this.loading = false;
        },
        error: (error) => {
          this.loading = false;
        },
      });
  }

  getFileIcon(fileType: string): string {
    if (fileType.startsWith('image/')) {
      return 'image';
    } else if (fileType.includes('pdf')) {
      return 'picture_as_pdf';
    } else if (fileType.includes('spreadsheet') || fileType.includes('excel')) {
      return 'table_chart';
    } else if (fileType.includes('document') || fileType.includes('word')) {
      return 'description';
    } else {
      return 'insert_drive_file';
    }
  }

  deleteUploadedFile(riskAttachmentId: string, index: number): void {
    this.commandDialogService.deleteConfirmtionDialog(this.translationService.getValue("ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_RISK_ATTACHMENT")).subscribe({
      next: (result: boolean) => {
        if (result) {
          this.sub$.sink = this.riskAttachmentService.deleteUploadedFile(riskAttachmentId).subscribe({
            next: () => {
              this.toastrService.success(this.translationService.getValue('FILE_DELETED_SUCCESSFULLY'));
              this.fileArray.splice(index, 1);
            },
            error: (error) => {
              this.toastrService.error(error.massage);
            },
          });
        }
      }
    });
  }

  saveRiskRequestAttachmentChunk(riskRequestAttachment: RiskAttachment): Observable<RiskAttachment | CommonError> {
    return this.riskAttachmentService.saveRiskAttachment(riskRequestAttachment);
  }

  uploadChunk(formData: FormData): Observable<RiskAttachmentChunk | CommonError> {
    return this.riskAttachmentService.uploadRiskAttachmentChunk(formData);
  }

  completeUpload(riskAttachmentId: string, status: boolean): Observable<any> {
    return this.riskAttachmentService.markAsRiskAttachmentChunkComplete(riskAttachmentId, status);
  }

  onMeidaView(document: RiskAttachment) {
    const documentView: DocumentView = {
      documentId: document.id,
      name: document.name,
      extension: document.extension,
      isVersion: false,
      isFromPublicPreview: false,
      isPreviewDownloadEnabled: true,
      isFileRequestDocument: false,
      totalChunk: document.totalChunk,
      moduleNo: QmsModuleEnum.Risk,
      isChunk: true
    };
    this.overlay.open(BasePreviewComponent, {
      position: 'center',
      origin: 'global',
      panelClass: ['file-preview-overlay-container', 'white-background'],
      data: documentView,
    });
  }
}
