import { Injectable, inject } from '@angular/core';
import { HttpClient } from '@angular/common/http';
import { Observable, timer } from 'rxjs';
import { catchError, retry } from 'rxjs/operators';
import { CommonHttpErrorService } from '@core/error-handler/common-http-error.service';
import { RiskAttachment } from '../model/risk-attachment';
import { CommonError } from '@core/error-handler/common-error';
import { RiskAttachmentChunk } from '../model/risk-attachment-chunk';

@Injectable({
  providedIn: 'root'
})
export class RiskAttachmentService {
  maxRetries: number = 2; // Maximum number of retries for chunk uploads
  private httpClient = inject(HttpClient);
  private commonHttpErrorService = inject(CommonHttpErrorService);

  saveRiskAttachment(response: RiskAttachment): Observable<RiskAttachment | CommonError> {
    const url = 'RiskAttachment';
    return this.httpClient
      .post<RiskAttachment>(url, response)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  uploadRiskAttachmentChunk(riskAttachmentChunk: FormData): Observable<RiskAttachmentChunk | CommonError> {
    const url = 'RiskAttachment/Chunk';
    return this.httpClient
      .post<RiskAttachmentChunk>(url, riskAttachmentChunk)
      .pipe(
        retry({
          count: this.maxRetries,
          delay: (error, retryCount) => {
            console.warn(`Retrying chunk... Attempt ${retryCount}`);
            return timer(Math.pow(2, retryCount) * 1000);
          },
        }),
        catchError(this.commonHttpErrorService.handleError)
      );
  }

  markAsRiskAttachmentChunkComplete(riskAttachmentChunkId: string, status: boolean): Observable<void | CommonError> {
    const url = 'RiskAttachment/Chunk/uploadStatus';
    return this.httpClient
      .post<void>(url, { riskAttachmentId: riskAttachmentChunkId, status: status })
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getAllRiskAttachmentsByRiskId(requestId: string): Observable<RiskAttachment[]> {
    const url = `RiskAttachment/${requestId}`;
    return this.httpClient
      .get<RiskAttachment[]>(url);

  }

  deleteUploadedFile(riskAttachmentId: string): Observable<void | CommonError> {
    const url = `RiskAttachment/${riskAttachmentId}`;
    return this.httpClient
      .delete<void>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }
}
