import { Component, Inject, inject, OnInit } from '@angular/core';
import { FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { BaseComponent } from '../../../base.component';
import { RiskManagementCategory } from '../../model/risk-category';
import { MAT_DIALOG_DATA, MatDialogModule, MatDialogRef } from '@angular/material/dialog';
import { MatButtonModule } from '@angular/material/button';
import { RiskManagementCategoryService } from '../risk-management-category.service';

import { ToastrService } from '@core/services/toastr-service';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { MatCardModule } from '@angular/material/card';
import { MatIconModule } from '@angular/material/icon';

@Component({
  selector: 'app-manage-risk-management-category',
  imports: [
    ReactiveFormsModule,
    MatDialogModule,
    MatButtonModule,
    PageHelpTextComponent,
    TranslateModule,
    MatCardModule,
    MatIconModule
  ],
  templateUrl: './manage-risk-category.component.html',
  styleUrl: './manage-risk-category.component.scss'
})
export class ManageRiskManagementCategoryComponent extends BaseComponent implements OnInit {
  categoryForm: FormGroup;
  fb = inject(FormBuilder);
  riskManagementCategoryService = inject(RiskManagementCategoryService);
  toastrService = inject(ToastrService);
  isEdit = false;

  constructor(
    private dialogRef: MatDialogRef<ManageRiskManagementCategoryComponent>,
    @Inject(MAT_DIALOG_DATA) public categoryDetail: RiskManagementCategory
  ) {
    super();
  }

  ngOnInit(): void {
    this.createCategoryForm();
    if (this.categoryDetail) {
      this.isEdit = true;
      this.categoryForm.patchValue(this.categoryDetail);
    }
  }

  createCategoryForm() {
    this.categoryForm = this.fb.group({
      id: [''],
      name: ['', [Validators.required]],
      description: [''],
    })
  }

  onSubmit() {
    if (!this.categoryForm.valid) {
      this.categoryForm.markAllAsTouched();
      return;
    }

    const category = this.categoryForm.getRawValue();

    if (this.isEdit) {
      this.riskManagementCategoryService.updateRiskManagementCategory(category).subscribe({
        next: (result) => {
          this.toastrService.success(this.translationService.getValue('RISK_MANAGEMENT_CATEGORY_UPDATED_SUCCESSFULLY'));
          this.dialogRef.close(result);
        },
      });
    } else {
      this.riskManagementCategoryService.addRiskManagementCategory(category).subscribe({
        next: (result) => {
          this.toastrService.success(this.translationService.getValue('RISK_MANAGEMENT_CATEGORY_ADDED_SUCCESSFULLY'));
          this.dialogRef.close(result);
        },
      });
    }
  }

  onCancel() {
    this.dialogRef.close();
  }
}
