import { Component, Inject, inject, OnInit, Optional } from '@angular/core';
import { RiskManagementControlListComponent } from "../risk-management-control-list/risk-management-control-list.component";
import { ActivatedRoute, RouterLink } from '@angular/router';
import { FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { MatSelectModule } from '@angular/material/select';
import { MatCardModule } from '@angular/material/card';
import { BaseComponent } from '../../../base.component';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { RISK_MANAGEMENT_CONTROL_STATUS } from '../../model/risk-control-status.enum';
import { UserStore } from '../../../user/store/user.store';
import { RISK_MANAGEMENT_CONTROL_EFFECTIVENESS } from '../../model/risk-control-effectiveness.enum';
import { TextEditorComponent } from '@shared/text-editor/text-editor.component';
import { MatButtonModule } from '@angular/material/button';
import { RISK_MANAGEMENT_CONTROL_TYPE } from '../../model/risk-control-type.enum';
import { RiskControl } from '../../model/risk-control';
import { RiskManagementControlService } from '../risk-management-control.service';

import { ToastrService } from '@core/services/toastr-service';
import { MAT_DIALOG_DATA, MatDialogRef } from '@angular/material/dialog';
import { NgClass } from '@angular/common';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { MatIconModule } from '@angular/material/icon';
import { RichTextRequired } from '@shared/validators/rich-text.validator';

@Component({
  selector: 'app-manage-risk-management-control',
  imports: [
    RiskManagementControlListComponent,
    RouterLink,
    ReactiveFormsModule,
    MatSelectModule,
    MatCardModule,
    MatDatepickerModule,
    TextEditorComponent,
    MatButtonModule,
    NgClass,
    HasClaimDirective,
    TranslateModule,
    MatIconModule
  ],
  templateUrl: './manage-risk-management-control.component.html',
  styleUrl: './manage-risk-management-control.component.scss'
})
export class ManageRiskManagementControlComponent extends BaseComponent implements OnInit {
  riskManagementControlForm: FormGroup;
  fb = inject(FormBuilder);
  minDueDate: Date = new Date();
  userStore = inject(UserStore);
  route = inject(ActivatedRoute);
  riskManagementControlService = inject(RiskManagementControlService);
  toastrService = inject(ToastrService);
  riskControlReaponse: RiskControl = {} as RiskControl;

  statusList = Object.keys(RISK_MANAGEMENT_CONTROL_STATUS)
    .filter(key => !isNaN(Number(RISK_MANAGEMENT_CONTROL_STATUS[key as any])))
    .map(key => ({
      label: key,
      value: RISK_MANAGEMENT_CONTROL_STATUS[key as keyof typeof RISK_MANAGEMENT_CONTROL_STATUS]
    }));
  effectivenessList = Object.keys(RISK_MANAGEMENT_CONTROL_EFFECTIVENESS)
    .filter(key => !isNaN(Number(RISK_MANAGEMENT_CONTROL_EFFECTIVENESS[key as any])))
    .map(key => ({
      label: key,
      value: RISK_MANAGEMENT_CONTROL_EFFECTIVENESS[key as keyof typeof RISK_MANAGEMENT_CONTROL_EFFECTIVENESS]
    }));
  controlTypeList = Object.keys(RISK_MANAGEMENT_CONTROL_TYPE)
    .filter(key => !isNaN(Number(RISK_MANAGEMENT_CONTROL_TYPE[key as any])))
    .map(key => ({
      label: key,
      value: RISK_MANAGEMENT_CONTROL_TYPE[key as keyof typeof RISK_MANAGEMENT_CONTROL_TYPE]
    }));

  constructor(
    @Optional() private dialogRef: MatDialogRef<ManageRiskManagementControlComponent>,
    @Optional() @Inject(MAT_DIALOG_DATA) public data: { riskManagementControl: RiskControl, isViewOnly: boolean },
  ) {
    super();
  }

  ngOnInit(): void {
    this.createRiskManagementControlForm();

    this.route.parent?.paramMap.subscribe(params => {
      this.riskManagementControlForm.patchValue({
        riskId: params.get('id'),
      });
    });

    if (this.data !== null) {
      this.riskManagementControlForm.patchValue(this.data.riskManagementControl);
    } else {
      this.data = {
        riskManagementControl: {} as RiskControl,
        isViewOnly: false
      };
    }
  }

  createRiskManagementControlForm() {
    this.riskManagementControlForm = this.fb.group({
      id: [''],
      riskId: [''],
      title: ['', [Validators.required]],
      description: ['', [RichTextRequired]],
      controlType: ['', [Validators.required]],
      status: ['', [Validators.required]],
      effectiveness: ['', [Validators.required]],
      responsibleId: ['', [Validators.required]],
      implementationDate: [this.minDueDate, [Validators.required]],
      reviewDate: [this.minDueDate],
    });
  }

  onSubmit() {
    if (!this.riskManagementControlForm.valid) {
      this.riskManagementControlForm.markAllAsTouched();
      return;
    }

    const controlData = this.riskManagementControlForm.getRawValue();

    if (this.data.riskManagementControl.id) {
      this.sub$.sink = this.riskManagementControlService.updateRiskManagementControl(controlData).subscribe({
        next: (response: RiskControl) => {
          this.toastrService.success(this.translationService.getValue('RISK_CONTROL_UPDATED_SUCCESSFULLY'));
          this.riskManagementControlForm.get('riskId')?.setValue(response.riskId);
          this.dialogRef.close(response);
        }
      });
    } else {
      this.sub$.sink = this.riskManagementControlService.addRiskManagementControl(controlData).subscribe({
        next: (response: RiskControl) => {
          this.toastrService.success(this.translationService.getValue('RISK_MANAGEMENT_CONTROL_CREATED_SUCCESSFULLY'));
          this.riskControlReaponse = response;
          this.riskManagementControlForm.reset();
          this.riskManagementControlForm.get('riskId')?.setValue(this.riskControlReaponse.riskId);
        }
      });
    }
  }

  onClose() {
    this.dialogRef.close();
  }
}
