import { Component, inject, Input, OnChanges, OnInit, SimpleChanges } from '@angular/core';
import { RiskControl } from '../../model/risk-control';
import { MatTableModule } from '@angular/material/table';
import { BaseComponent } from '../../../base.component';
import { RiskManagementControlService } from '../risk-management-control.service';

import { ToastrService } from '@core/services/toastr-service';
import { MatDialog } from '@angular/material/dialog';
import { ManageRiskManagementControlComponent } from '../manage-risk-management-control/manage-risk-management-control.component';
import { TranslateModule } from '@ngx-translate/core';
import { LimitToPipe } from '@shared/pipes/limit-to.pipe';
import { ActivatedRoute } from '@angular/router';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { RiskControlStatusPipe } from '../../pipes/risk-control-status.pipe';
import { RiskControlEffectivenessPipe } from '../../pipes/risk-control-effectiveness.pipe';
import { NgClass } from '@angular/common';
import { RiskControlTypePipe } from '../../pipes/risk-control-type.pipe';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';
import { MatButtonModule } from '@angular/material/button';

@Component({
  selector: 'app-risk-management-control-list',
  imports: [
    MatTableModule,
    TranslateModule,
    LimitToPipe,
    UTCToLocalTime,
    RiskControlStatusPipe,
    RiskControlEffectivenessPipe,
    RiskControlTypePipe,
    NgClass,
    MatIconModule,
    MatCardModule,
    MatButtonModule
  ],
  templateUrl: './risk-management-control-list.component.html',
  styleUrl: './risk-management-control-list.component.scss'
})
export class RiskManagementControlListComponent extends BaseComponent implements OnInit, OnChanges {
  @Input() riskManagementControlReaponse: RiskControl = {} as RiskControl;

  displayedColumns: string[] = ['action', 'title', 'controlType', 'status', 'effectiveness', 'responsibleId', 'implementationDate', 'reviewDate'];
  riskManagementControlList: RiskControl[] = [];
  riskManagementControlService = inject(RiskManagementControlService);
  toastrService = inject(ToastrService);
  dialog = inject(MatDialog);
  route = inject(ActivatedRoute);

  ngOnInit(): void {
    this.route.parent?.paramMap.subscribe(params => {
      const riskManagementControlId = params.get('id');
      if (riskManagementControlId) {
        this.getAllControls(riskManagementControlId);
      }
    });
  }

  ngOnChanges(changes: SimpleChanges): void {
    if (changes['riskManagementControlReaponse'] && this.riskManagementControlReaponse) {
      this.riskManagementControlList = [...this.riskManagementControlList, this.riskManagementControlReaponse];
    }
  }

  getAllControls(riskId: string) {
    this.sub$.sink = this.riskManagementControlService.getRiskManagementControlList(riskId).subscribe({
      next: (controls) => {
        const riskManagementControls = controls as RiskControl[];
        if (riskManagementControls && riskManagementControls.length > 0) {
          this.riskManagementControlList = riskManagementControls;
        } else {
          this.riskManagementControlList = [];
        }
      },
      error: (error) => {
        this.toastrService.error(error.message);
      }
    });
  }

  viewRiskManagementControl(riskManagementControl: RiskControl) {
    const dialogRef = this.dialog.open(ManageRiskManagementControlComponent, {
      maxWidth: '70vw',
      width: '100%',
      data: { riskManagementControl, isViewOnly: true },
    });

    this.sub$.sink = dialogRef.afterClosed().subscribe((result: RiskControl) => {
      if (result) {
        const index = this.riskManagementControlList.findIndex(a => a.id === result.id);
        if (index !== -1) {
          this.riskManagementControlList[index] = result;
          this.riskManagementControlList = [...this.riskManagementControlList];
        }
      }
    });
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.riskManagementControlList.indexOf(row);
  }
}
