import { AfterViewInit, Component, inject, OnInit, ViewChild } from '@angular/core';
import { MatPaginator } from '@angular/material/paginator';
import { MatSort, MatSortModule } from '@angular/material/sort';
import { PageSizeOption } from '@core/utils/global-config';

import { ToastrService } from '@core/services/toastr-service';
import { Subject, debounceTime, distinctUntilChanged, merge, tap } from 'rxjs';
import { BaseComponent } from '../../base.component';
import { UserStore } from '../../user/store/user.store';
import { RiskManagementCategory } from '../model/risk-category';
import { RISK_MANAGEMENT_CONTROL_STATUS } from '../model/risk-control-status.enum';
import { RiskManagementCategoryService } from '../risk-category/risk-management-category.service';
import { Risk } from '../model/risk';
import { RiskResource } from '../model/risk-resource';
import { RiskLogService } from './risk-log.service';
import { HttpResponse } from '@angular/common/http';
import { MatTableModule } from '@angular/material/table';
import { MatSelectModule } from '@angular/material/select';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { FormsModule } from '@angular/forms';
import { RISK_LOG_STATUS } from '../model/risk-log-status.enum';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { NgClass } from '@angular/common';
import { RiskControlStatusPipe } from '../pipes/risk-control-status.pipe';
import { RiskLevalPipe } from '../pipes/risk-level.pipe';
import { RiskLogStatusPipe } from '../pipes/risk-log-status.pipe';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-risk-log-list',
  imports: [
    MatTableModule,
    MatSelectModule,
    MatPaginator,
    MatSortModule,
    MatDatepickerModule,
    FormsModule,
    PageHelpTextComponent,
    TranslateModule,
    UTCToLocalTime,
    RiskControlStatusPipe,
    RiskLevalPipe,
    RiskLogStatusPipe,
    NgClass,
    MatIconModule,
    MatCardModule
  ],
  templateUrl: './risk-log-list.component.html',
  styleUrl: './risk-log-list.component.scss'
})
export class RiskLogListComponent extends BaseComponent implements OnInit, AfterViewInit {
  displayedColumns: string[] = ['title', 'createdDate', 'assignToId', 'riskCategoryId', 'status', 'riskScore', 'riskLavel', 'createdBy', 'logStatus'];
  displayedColumnSecondary: string[] = ['search-title', 'search-createdDate', 'search-assignToId', 'search-riskCategoryId', 'search-status', 'search-riskScore', 'search-riskLavel', 'search-createdBy', 'search-logStatus'];
  footerToDisplayedColumns: string[] = ['footer'];
  riskLavel: string[] = ['Critical', 'High', 'Low', 'Very High', 'Medium'];
  riskLogService = inject(RiskLogService);
  userStore = inject(UserStore);
  toastr = inject(ToastrService);
  riskCategoryService = inject(RiskManagementCategoryService);
  riskCateGoryList: RiskManagementCategory[] = [];
  pageOption = PageSizeOption;
  riskLogList: Risk[] = [];

  @ViewChild(MatPaginator) paginator: MatPaginator;
  @ViewChild(MatSort) sort: MatSort;

  statusList = Object.keys(RISK_MANAGEMENT_CONTROL_STATUS)
    .filter(key => !isNaN(Number(RISK_MANAGEMENT_CONTROL_STATUS[key as any])))
    .map(key => ({
      name: key,
      id: RISK_MANAGEMENT_CONTROL_STATUS[key as keyof typeof RISK_MANAGEMENT_CONTROL_STATUS]
    }));
  logStatus = Object.keys(RISK_LOG_STATUS)
    .filter((key) => !isNaN(Number(RISK_LOG_STATUS[key as any])))
    .map((key) => ({
      name: key,
      id: RISK_LOG_STATUS[key as keyof typeof RISK_LOG_STATUS],
    }));


  filterParameter$: Subject<string> = new Subject<string>();
  riskSearchParameters: RiskResource = {
    title: '',
    createdFromDate: undefined,
    createdToDate: undefined,
    assignedToId: '',
    riskCategoryId: '',
    status: undefined,
    riskScore: undefined,
    riskLevel: '',
    orderBy: 'createdDate desc',
    pageSize: 10,
    skip: 0,
    totalCount: 0,
    logStatus: undefined,
  };
  _titleFilter = this.riskSearchParameters.title;
  _createdFromDateFilter = this.riskSearchParameters.createdFromDate;
  _createdToDateFilter = this.riskSearchParameters.createdToDate;
  _assignedToFilter = this.riskSearchParameters.assignedToId;
  _riskCategoryFilter = this.riskSearchParameters.riskCategoryId;
  _statusFilter = this.riskSearchParameters.status;
  _riskScoreFilter = this.riskSearchParameters.riskScore;
  _riskLevelFilter = this.riskSearchParameters.riskLevel;
  _logStatusFilter: number | null = null;

  public get titleFilter(): string {
    return this._titleFilter;
  }

  public set titleFilter(v: string) {
    if (this._titleFilter !== v) {
      this._titleFilter = v;
      const titleFilter = `title#${v}`;
      this.filterParameter$.next(titleFilter);
    }
  }

  public get createdFromDateFilter(): Date | undefined | null {
    return this._createdFromDateFilter;
  }

  public set createdFromDateFilter(v: Date | null) {
    if (this._createdFromDateFilter !== v) {
      this._createdFromDateFilter = v;
      this.riskSearchParameters.createdFromDate = v;
      this.filterParameter$.next(`createdFromDate#${v}`);
    }
  }

  public get createdToDateFilter(): Date | undefined | null {
    return this._createdToDateFilter;
  }

  public set createdToDateFilter(v: Date) {
    if (this._createdToDateFilter !== v) {
      this._createdToDateFilter = v;
      this.riskSearchParameters.createdToDate = v;
      this.filterParameter$.next(`createdToDate#${v}`);
    }
  }

  public get assignedToFilter(): string {
    return this._assignedToFilter;
  }

  public set assignedToFilter(v: string) {
    if (this._assignedToFilter !== v) {
      this._assignedToFilter = v;
      this.riskSearchParameters.assignedToId = v;
      this.filterParameter$.next(`assignedToId#${v}`);
    }
  }

  public get riskCategoryFilter(): string {
    return this._riskCategoryFilter;
  }

  public set riskCategoryFilter(v: string) {
    if (this._riskCategoryFilter !== v) {
      this._riskCategoryFilter = v;
      this.riskSearchParameters.riskCategoryId = v;
      this.filterParameter$.next(`riskCategoryId#${v}`);
    }
  }

  public get statusFilter(): number {
    return this._statusFilter ?? 0;
  }

  public set statusFilter(v: number) {
    if (this._statusFilter !== v) {
      this._statusFilter = v;
      this.riskSearchParameters.status = v;
      this.filterParameter$.next(`status#${v}`);
    }
  }

  public get riskScoreFilter(): number | null {
    return this._riskScoreFilter === undefined ? null : this._riskScoreFilter;
  }

  public set riskScoreFilter(v: number | null) {
    if (this._riskScoreFilter !== v) {
      this._riskScoreFilter = (v === null || isNaN(v as number)) ? undefined : v as number;
      this.riskSearchParameters.riskScore = this._riskScoreFilter;
      this.filterParameter$.next(`riskScore#${this._riskScoreFilter}`);
    }
  }

  public get riskLevelFilter(): string {
    return this._riskLevelFilter ?? '';
  }

  public set riskLevelFilter(v: string) {
    if (this._riskLevelFilter !== v) {
      this._riskLevelFilter = v;
      this.riskSearchParameters.riskLevel = v;
      this.filterParameter$.next(`riskLevel#${v}`);
    }
  }

  public get logStatusFilter(): number | null {
    return this._logStatusFilter;
  }

  public set logStatusFilter(v: number | null) {
    if (this._logStatusFilter !== v) {
      this._logStatusFilter = isNaN(v as number) ? null : v;
      this.riskSearchParameters.logStatus = this._logStatusFilter === null ? undefined : this._logStatusFilter;
      this.filterParameter$.next(`logStatus#${this._logStatusFilter}`);
    }
  }

  ngOnInit() {
    this.getRiskCategoryList();
    this.reFresh();

    this.sub$.sink = this.filterParameter$
      .pipe(debounceTime(1000), distinctUntilChanged())
      .subscribe((c: string) => {
        this.riskSearchParameters.skip = 0;
        this.paginator.pageIndex = 0;
        const filterArray: Array<string> = c.split('#');
        if (filterArray[0] === 'title') {
          this.riskSearchParameters.title = filterArray[1];
        } else if (filterArray[0] === 'createdFromDate') {
          if (filterArray[1] != 'null') {
            this.riskSearchParameters.createdFromDate = new Date(
              filterArray[1]
            );
            this.riskSearchParameters.createdToDate = this.createdToDateFilter;
          } else {
            this.riskSearchParameters.createdFromDate = undefined;
            this.riskSearchParameters.createdToDate = undefined;
          }
        } else if (filterArray[0] === 'createdToDate') {
          if (filterArray[1] != 'null') {
            this.riskSearchParameters.createdToDate = new Date(
              filterArray[1]
            );
            this.riskSearchParameters.createdFromDate =
              this.createdFromDateFilter;
          } else {
            this.riskSearchParameters.createdFromDate = undefined;
            this.riskSearchParameters.createdToDate = undefined;
          }
        } else if (filterArray[0] === 'assignedToId') {
          this.riskSearchParameters.assignedToId = filterArray[1];
        } else if (filterArray[0] === 'riskCategoryId') {
          this.riskSearchParameters.riskCategoryId = filterArray[1];
        } else if (filterArray[0] === 'status') {
          this.riskSearchParameters.status = filterArray[1] ? parseInt(filterArray[1], 10) : undefined;
        } else if (filterArray[0] === 'riskScore') {
          const score = filterArray[1] ? parseInt(filterArray[1], 10) : undefined;
          if (score !== undefined && score > 25) {
            this.toastr.error('Risk score must not exceed 25');
            return;
          }
          this.riskSearchParameters.riskScore = score;
        } else if (filterArray[0] === 'riskLevel') {
          this.riskSearchParameters.riskLevel = filterArray[1];
        } else if (filterArray[0] === 'logStatus') {
          this.riskSearchParameters.logStatus = filterArray[1] ? parseInt(filterArray[1], 10) : undefined;
        }
        this.reFresh();
      });
  }

  clearOrderDates() {
    this._createdFromDateFilter = undefined;
    this._createdToDateFilter = undefined;
  }

  getRiskCategoryList() {
    this.sub$.sink = this.riskCategoryService.getRiskManagementCategory().subscribe({
      next: (data) => {
        const riskCategory = data as RiskManagementCategory[];
        if (riskCategory) {
          this.riskCateGoryList = [...riskCategory];
        }
        return;
      },
    });
  }

  reFresh() {
    this.loadRiskList(this.riskSearchParameters)
  }

  ngAfterViewInit(): void {
    this.sub$.sink = this.sort.sortChange.subscribe(() => {
      this.paginator.pageIndex = 0;
    });

    this.sub$.sink = merge(this.sort.sortChange, this.paginator.page)
      .pipe(
        tap(() => {
          this.riskSearchParameters.skip = this.paginator.pageIndex * this.paginator.pageSize;
          this.riskSearchParameters.pageSize = this.paginator.pageSize;
          this.riskSearchParameters.orderBy = this.sort.active + ' ' + this.sort.direction;
          this.reFresh();
        })
      ).subscribe();
  }

  loadRiskList(filter: RiskResource) {
    this.sub$.sink = this.riskLogService.getRisks(this.riskSearchParameters).subscribe({
      next: (httpResponse: HttpResponse<Risk[]>) => {
        if (httpResponse && httpResponse.body) {
          this.riskLogList = [...httpResponse.body as Risk[]];
        }
        const paginationHeader = httpResponse.headers.get('x-pagination');
        if (paginationHeader) {
          const pagination = JSON.parse(paginationHeader);
          this.riskSearchParameters = { ...filter, totalCount: pagination.totalCount };
        }
      }
    })
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.riskLogList.indexOf(row);
  }
}
