import { Component, Inject, inject, OnInit, Optional } from '@angular/core';
import { RiskManagementReviewListComponent } from "../risk-management-review-list/risk-management-review-list.component";
import { ActivatedRoute, RouterLink } from '@angular/router';
import { TextEditorComponent } from "../../../shared/text-editor/text-editor.component";
import { FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { MatSelectModule } from '@angular/material/select';
import { MatCardModule } from '@angular/material/card';
import { BaseComponent } from '../../../base.component';
import { UserStore } from '../../../user/store/user.store';
import { MatButtonModule } from '@angular/material/button';
import { RiskReview } from '../../model/risk-review';
import { RiskManagementReviewService } from '../risk-management-review.service';

import { ToastrService } from '@core/services/toastr-service';
import { MAT_DIALOG_DATA, MatDialogRef } from '@angular/material/dialog';
import { NgClass } from '@angular/common';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { MatIconModule } from '@angular/material/icon';

@Component({
  selector: 'app-manage-risk-management-review',
  imports: [
    RiskManagementReviewListComponent,
    RouterLink,
    TextEditorComponent,
    MatDatepickerModule,
    MatSelectModule,
    MatCardModule,
    ReactiveFormsModule,
    MatButtonModule,
    NgClass,
    HasClaimDirective,
    TranslateModule,
    MatIconModule
  ],
  templateUrl: './manage-risk-review.component.html',
  styleUrl: './manage-risk-review.component.scss'
})
export class ManageRiskManagementReviewComponent extends BaseComponent implements OnInit {
  riskManagementReviewForm: FormGroup;
  fb = inject(FormBuilder);
  minDueDate: Date = new Date();
  userStore = inject(UserStore);
  riskReviewResponse: RiskReview = {} as RiskReview;
  route = inject(ActivatedRoute);
  riskReviewService = inject(RiskManagementReviewService);
  toastrService = inject(ToastrService);

  constructor(
    @Optional() private dialogRef: MatDialogRef<ManageRiskManagementReviewComponent>,
    @Optional() @Inject(MAT_DIALOG_DATA) public data: { riskReview: RiskReview, isViewOnly: boolean },
  ) {
    super();
  }

  ngOnInit(): void {
    this.createRiskManagementReviewForm();

    this.sub$.sink = this.route.parent?.params.subscribe(params => {
      const riskId = params['id'];
      if (riskId) {
        this.riskManagementReviewForm.get('riskId')?.setValue(riskId);
      }
    });

    if (this.data !== null) {
      this.riskManagementReviewForm.patchValue(this.data.riskReview);
    } else {
      this.data = {
        riskReview: {} as RiskReview,
        isViewOnly: false
      }
    }
  }

  createRiskManagementReviewForm() {
    this.riskManagementReviewForm = this.fb.group({
      id: [''],
      reviewerId: [''],
      riskId: [''],
      likelihood: ['', [Validators.required]],
      impact: ['', [Validators.required]],
      reviewDate: [this.minDueDate, [Validators.required]],
      reviewerComments: ['', [Validators.required]],
    });
  }

  onSubmit() {
    if (!this.riskManagementReviewForm.valid) {
      this.riskManagementReviewForm.markAllAsTouched();
      return;
    }

    const formValue = this.riskManagementReviewForm.getRawValue();

    if (this.data.riskReview && this.data.riskReview.id) {
      this.sub$.sink = this.riskReviewService.updateRiskManagementReview(formValue).subscribe({
        next: (response: RiskReview) => {
          this.toastrService.success(this.translationService.getValue('RISK_REVIEW_UPDATED_SUCCESSFULLY'));
          this.riskManagementReviewForm.get('riskId')?.setValue(response.riskId);
          this.dialogRef.close(response);
        }
      });
    } else {
      this.sub$.sink = this.riskReviewService.addRiskManagementReview(formValue).subscribe({
        next: (response: RiskReview) => {
          this.riskReviewResponse = response;
          this.toastrService.success(this.translationService.getValue('RISK_REVIEW_CRATED_SUCCESSFULLY'));
          this.riskManagementReviewForm.reset();
          this.riskManagementReviewForm.get('riskId')?.setValue(response.riskId);
        }
      });
    }
  }

  onNoClick(): void {
    this.dialogRef.close();
  }
}
