import { NgClass } from '@angular/common';
import { Component, inject, OnInit } from '@angular/core';
import { FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { MatCardModule } from '@angular/material/card';
import { MatSelectModule } from '@angular/material/select';
import { TextEditorComponent } from '@shared/text-editor/text-editor.component';
import { BaseComponent } from '../../../base.component';
import { UserStore } from '../../../user/store/user.store';
import { RISK_MANAGEMENT_CONTROL_STATUS } from '../../model/risk-control-status.enum';
import { ActivatedRoute, Data, Router, RouterLink } from '@angular/router';
import { RiskManagementCategoryService } from '../../risk-category/risk-management-category.service';
import { RiskManagementCategory } from '../../model/risk-category';

import { ToastrService } from '@core/services/toastr-service';
import { RiskStore } from '../risk.store';
import { toObservable } from '@angular/core/rxjs-interop';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { MatButtonModule } from '@angular/material/button';
import { MatIconModule } from '@angular/material/icon';
import { MatDialog } from '@angular/material/dialog';
import { ManageRiskManagementCategoryComponent } from '../../risk-category/manage-risk-category/manage-risk-category.component';
import { HasClaimDirective } from '@shared/has-claim.directive';

@Component({
  selector: 'app-manage-risk',
  imports: [
    MatSelectModule,
    NgClass,
    TextEditorComponent,
    ReactiveFormsModule,
    MatCardModule,
    RouterLink,
    PageHelpTextComponent,
    TranslateModule,
    MatButtonModule,
    MatIconModule,
    HasClaimDirective
  ],
  templateUrl: './manage-risk.component.html',
  styleUrl: './manage-risk.component.scss'
})
export class ManageRiskComponent extends BaseComponent implements OnInit {
  isDetailsMode: boolean = true;
  riskForm: FormGroup;
  riskCateGoryList: RiskManagementCategory[] = [];
  fb = inject(FormBuilder);
  isEditMode: boolean = false;
  userStore = inject(UserStore);
  riskCategoryService = inject(RiskManagementCategoryService);
  riskStore = inject(RiskStore);
  toastrService = inject(ToastrService);
  router = inject(Router);
  route = inject(ActivatedRoute);
  dialog = inject(MatDialog);
  statusList = Object.keys(RISK_MANAGEMENT_CONTROL_STATUS)
    .filter(key => !isNaN(Number(RISK_MANAGEMENT_CONTROL_STATUS[key as any])))
    .map(key => ({
      label: key,
      value: RISK_MANAGEMENT_CONTROL_STATUS[key as keyof typeof RISK_MANAGEMENT_CONTROL_STATUS]
    }));

  constructor() {
    super();
    this.subscribeIsAddorUpdate();
  }

  ngOnInit(): void {
    this.createRiskForm();
    this.getRiskCategoryList();

    this.route.data.subscribe((data: Data) => {
      if (data['riskDetails']) {
        this.isEditMode = true;
        this.riskForm.patchValue(data['riskDetails']);
      } else {
        this.isEditMode = false;
      }
    });

    this.route.queryParams.subscribe((params) => {
      if (params['isDetailsMode'] == 'true') {
        this.isDetailsMode = false;
      }
    });
  }

  createRiskForm() {
    this.riskForm = this.fb.group({
      id: [''],
      title: ['', [Validators.required]],
      assignedToId: ['', [Validators.required]],
      riskCategoryId: ['', [Validators.required]],
      likelihood: ['', [Validators.required]],
      severity: ['', [Validators.required]],
      status: ['', [Validators.required]],
      description: [''],
    })
  }

  getRiskCategoryList() {
    this.sub$.sink = this.riskCategoryService.getRiskManagementCategory().subscribe({
      next: (data) => {
        const riskCategory = data as RiskManagementCategory[];
        if (riskCategory) {
          this.riskCateGoryList = [...riskCategory];
        }
      },
    });
  }

  onSubmit() {
    if (!this.riskForm.valid) {
      this.riskForm.markAllAsTouched();
      return;
    }

    const riskData = this.riskForm.getRawValue();

    if (this.isEditMode) {
      this.riskStore.updateRisk(riskData)
    } else {
      this.riskStore.addRisk(riskData);
    }
  }

  subscribeIsAddorUpdate() {
    toObservable(this.riskStore.isAddUpdate).subscribe((flag) => {
      if (flag) {
        const id = this.riskForm.get('id')?.value;
        const route = this.isDetailsMode ? ['/risk/list'] : [`/risk/details/${id}/control`];
        this.router.navigate(route);
      }
      this.riskStore.resetflag();
    });
  }

  addRiskCategory(): void {
    const dialogRef = this.dialog.open(ManageRiskManagementCategoryComponent, {
      width: '500px',
    });
    this.sub$.sink = dialogRef.afterClosed().subscribe((result: RiskManagementCategory) => {
      if (result) {
        this.riskCateGoryList = [...this.riskCateGoryList, result];
        this.riskForm.patchValue({ riskCategoryId: result.id });
      }
    });
  }
}
