import { NgClass } from '@angular/common';
import { Component, inject, OnInit } from '@angular/core';
import { FormsModule } from '@angular/forms';
import { MatButtonModule } from '@angular/material/button';
import { MatCardModule } from '@angular/material/card';
import { MatIconModule } from '@angular/material/icon';
import { TranslateModule } from '@ngx-translate/core';
import { PageHelpTextComponent } from "@shared/page-help-text/page-help-text.component";
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { TextEditorComponent } from '@shared/text-editor/text-editor.component';
import { Risk } from '../../model/risk';
import { RiskControlStatusPipe } from "../../pipes/risk-control-status.pipe";
import { RISK_MANAGEMENT_CONTROL_STATUS } from '../../model/risk-control-status.enum';
import { ToastrService } from '@core/services/toastr-service';
import { TranslationService } from '@core/services/translation.service';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { RiskControl } from '../../model/risk-control';
import { BaseComponent } from '../../../base.component';
import { RiskManagementControlService } from '../../risk-control/risk-management-control.service';
import { RiskControlTypePipe } from "../../pipes/risk-control-type.pipe";
import { RiskService } from '../risk.service';
import { RISK_MANAGEMENT_CONTROL_TYPE } from '../../model/risk-control-type.enum';
import { RISK_LEVEL } from '../../model/risk-level.enum';
import { ManageRiskAttachmentComponent } from "../../risk-attachment/manage-risk-attachment/manage-risk-attachment.component";
import { RiskControlEffectivenessPipe } from "../../pipes/risk-control-effectiveness.pipe";
import { RISK_MANAGEMENT_CONTROL_EFFECTIVENESS } from '../../model/risk-control-effectiveness.enum';
import { RiskManagementReviewService } from '../../risk-review/risk-management-review.service';

@Component({
  selector: 'app-my-risk-list',
  imports: [
    PageHelpTextComponent,
    MatCardModule,
    MatButtonModule,
    MatIconModule,
    TranslateModule,
    FormsModule,
    UTCToLocalTime,
    RiskControlStatusPipe,
    TextEditorComponent,
    RiskControlTypePipe,
    NgClass,
    ManageRiskAttachmentComponent,
    RiskControlEffectivenessPipe
],
  standalone: true,
  templateUrl: './my-risk-list.html',
  styleUrl: './my-risk-list.scss'
})
export class MyRiskList extends BaseComponent implements OnInit {
  toastrService = inject(ToastrService);
  translateService = inject(TranslationService);
  commonDialogService = inject(CommonDialogService);
  riskControlService = inject(RiskManagementControlService);
  riskReviewService = inject(RiskManagementReviewService);
  riskService = inject(RiskService);
  RISK_MANAGEMENT_CONTROL_STATUS = RISK_MANAGEMENT_CONTROL_STATUS;
  RISK_MANAGEMENT_CONTROL_TYPE = RISK_MANAGEMENT_CONTROL_TYPE;
  RISK_LEVEL = RISK_LEVEL;
  assignedRisks: Risk[] = [];
  expandedIndex: number | null = null;
  riskId: string = '';
  RISK_CONTROL_EFFECTIVENESS = RISK_MANAGEMENT_CONTROL_EFFECTIVENESS;

  ngOnInit(): void {
    this.loadUserRisks();
  }

  loadUserRisks(): void {
    this.sub$.sink = this.riskService.getUserRiskControls().subscribe({
      next: (risks: Risk[] | any) => {
        const assignedRisks = risks as Risk[];
        if (assignedRisks) {
          this.assignedRisks = assignedRisks;
        }
      }
    });
  }

  updateStatus(control: RiskControl, newStatus: number): void {
    const updatedControl: RiskControl = { ...control, status: newStatus };

    this.sub$.sink = this.commonDialogService
      .deleteConfirmtionDialog(`${this.translateService.getValue('ARE_YOU_SURE_YOU_WANT_TO_UPDATE_STATUS')}`)
      .subscribe((isTrue: boolean) => {
        if (isTrue) {
          this.riskControlService.updateRiskManagementControl(updatedControl).subscribe({
            next: () => {
              control.status = newStatus;
              this.toastrService.success(this.translateService.getValue('RISK_CONTROL_UPDATED_SUCCESSFULLY'));
            }
          });
        }
      });
  }

  updateReview(review: any): void {
    const updatedReview: any = { ...review };
    this.riskReviewService.updateRiskManagementReview(updatedReview).subscribe({
      next: () => {
        review.reviewerComments = updatedReview.reviewerComments;
        this.toastrService.success(this.translateService.getValue('RISK_REVIEW_UPDATED_SUCCESSFULLY'));
      }
    });
  }

  onAccordionToggle(index: number, riskId: string) {
    this.riskId = riskId;
    this.expandedIndex = this.expandedIndex === index ? null : index;
  }

  onFilesChanged(file: any) {
    this.assignedRisks.forEach(risk => {
      if (risk.id === this.riskId) {
        risk.riskAttachments = [...risk.riskAttachments || [], ...file];
      }
    });
  }
}
