import { AfterViewInit, Component, inject, OnInit, ViewChild } from '@angular/core';
import { MatPaginator } from '@angular/material/paginator';
import { MatSort, MatSortModule } from '@angular/material/sort';
import { MatTableModule } from '@angular/material/table';
import { RouterLink } from '@angular/router';
import { RiskStore } from '../risk.store';
import { BaseComponent } from '../../../base.component';
import { PageSizeOption } from '@core/utils/global-config';
import { MatSelectModule } from '@angular/material/select';
import { FormsModule } from '@angular/forms';
import { UserStore } from '../../../user/store/user.store';
import { RiskManagementCategoryService } from '../../risk-category/risk-management-category.service';
import { RiskManagementCategory } from '../../model/risk-category';
import { RISK_MANAGEMENT_CONTROL_STATUS } from '../../model/risk-control-status.enum';
import { debounceTime, distinctUntilChanged, merge, Subject, tap } from 'rxjs';
import { MatDatepickerModule } from '@angular/material/datepicker';

import { ToastrService } from '@core/services/toastr-service';
import { MatMenuModule } from '@angular/material/menu';
import { MatIconModule } from '@angular/material/icon';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { MatButtonModule } from '@angular/material/button';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { NgClass } from '@angular/common';
import { RiskLevalPipe } from '../../pipes/risk-level.pipe';
import { RiskControlStatusPipe } from '../../pipes/risk-control-status.pipe';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-risk-list',
  imports: [
    MatTableModule,
    RouterLink,
    MatSortModule,
    MatPaginator,
    MatSelectModule,
    FormsModule,
    MatDatepickerModule,
    MatMenuModule,
    MatIconModule,
    MatButtonModule,
    PageHelpTextComponent,
    HasClaimDirective,
    TranslateModule,
    UTCToLocalTime,
    NgClass,
    RiskLevalPipe,
    RiskControlStatusPipe,
    MatCardModule
  ],
  templateUrl: './risk-list.component.html',
  styleUrl: './risk-list.component.scss'
})
export class RiskListComponent extends BaseComponent implements OnInit, AfterViewInit {
  displayedColumns: string[] = ['action', 'title', 'createdDate', 'assignToId', 'riskCategoryId', 'status', 'riskScore', 'riskLavel', 'createdBy'];
  displayedColumnSecondary: string[] = ['search-action', 'search-title', 'search-createdDate', 'search-assignToId', 'search-riskCategoryId', 'search-status', 'search-riskScore', 'search-riskLavel', 'search-createdBy'];
  footerToDisplayedColumns: string[] = ['footer'];
  riskLavel: string[] = ['Critical', 'High', 'Low', 'Very High', 'Medium'];
  riskStore = inject(RiskStore);
  userStore = inject(UserStore);
  toastr = inject(ToastrService);
  riskCategoryService = inject(RiskManagementCategoryService);
  riskCateGoryList: RiskManagementCategory[] = [];
  pageOption = PageSizeOption;
  CommonDialogService = inject(CommonDialogService);

  @ViewChild(MatPaginator) paginator: MatPaginator;
  @ViewChild(MatSort) sort: MatSort;

  statusList = Object.keys(RISK_MANAGEMENT_CONTROL_STATUS)
    .filter(key => !isNaN(Number(RISK_MANAGEMENT_CONTROL_STATUS[key as any])))
    .map(key => ({
      name: key,
      id: RISK_MANAGEMENT_CONTROL_STATUS[key as keyof typeof RISK_MANAGEMENT_CONTROL_STATUS]
    }));

  filterParameter$: Subject<string> = new Subject<string>();
  riskSearchParameters = this.riskStore.filterParameters();
  _titleFilter = this.riskSearchParameters.title;
  _createdFromDateFilter = this.riskSearchParameters.createdFromDate;
  _createdToDateFilter = this.riskSearchParameters.createdToDate;
  _assignedToFilter = this.riskSearchParameters.assignedToId;
  _riskCategoryFilter = this.riskSearchParameters.riskCategoryId;
  _statusFilter = this.riskSearchParameters.status;
  _riskScoreFilter = this.riskSearchParameters.riskScore;
  _riskLevelFilter = this.riskSearchParameters.riskLevel;

  public get titleFilter(): string {
    return this._titleFilter;
  }

  public set titleFilter(v: string) {
    if (this._titleFilter !== v) {
      this._titleFilter = v;
      const titleFilter = `title#${v}`;
      this.filterParameter$.next(titleFilter);
    }
  }

  public get createdFromDateFilter(): Date | null {
    return this._createdFromDateFilter !== undefined ? this._createdFromDateFilter : null;
  }

  public set createdFromDateFilter(v: Date | null) {
    if (this._createdFromDateFilter !== v) {
      this._createdFromDateFilter = v;
      this.riskSearchParameters.createdFromDate = v;
      this.filterParameter$.next(`createdFromDate#${v}`);
    }
  }

  public get createdToDateFilter(): Date | null {
    return this._createdToDateFilter !== undefined ? this._createdToDateFilter : null;
  }

  public set createdToDateFilter(v: Date | null) {
    if (this._createdToDateFilter !== v) {
      this._createdToDateFilter = v;
      this.riskSearchParameters.createdToDate = v;
      this.filterParameter$.next(`createdToDate#${v}`);
    }
  }

  public get assignedToFilter(): string {
    return this._assignedToFilter;
  }

  public set assignedToFilter(v: string) {
    if (this._assignedToFilter !== v) {
      this._assignedToFilter = v;
      this.riskSearchParameters.assignedToId = v;
      this.filterParameter$.next(`assignedToId#${v}`);
    }
  }

  public get riskCategoryFilter(): string {
    return this._riskCategoryFilter;
  }

  public set riskCategoryFilter(v: string) {
    if (this._riskCategoryFilter !== v) {
      this._riskCategoryFilter = v;
      this.riskSearchParameters.riskCategoryId = v;
      this.filterParameter$.next(`riskCategoryId#${v}`);
    }
  }

  public get statusFilter(): number {
    return this._statusFilter ?? 0;
  }

  public set statusFilter(v: number) {
    if (this._statusFilter !== v) {
      this._statusFilter = v;
      this.riskSearchParameters.status = v;
      this.filterParameter$.next(`status#${v}`);
    }
  }

  public get riskScoreFilter(): number | null {
    return this._riskScoreFilter !== undefined ? this._riskScoreFilter : null;
  }

  public set riskScoreFilter(v: number | null) {
    if (this._riskScoreFilter !== v) {
      this._riskScoreFilter = isNaN(v as number) || v === null ? undefined : v;
      this.riskSearchParameters.riskScore = this._riskScoreFilter;
      this.filterParameter$.next(`riskScore#${this._riskScoreFilter}`);
    }
  }

  public get riskLevelFilter(): string {
    return this._riskLevelFilter ?? '';
  }

  public set riskLevelFilter(v: string) {
    if (this._riskLevelFilter !== v) {
      this._riskLevelFilter = v;
      this.riskSearchParameters.riskLevel = v;
      this.filterParameter$.next(`riskLevel#${v}`);
    }
  }

  ngOnInit() {
    this.getRiskCategoryList();

    this.sub$.sink = this.filterParameter$
      .pipe(debounceTime(1000), distinctUntilChanged())
      .subscribe((c: string) => {
        this.riskSearchParameters.skip = 0;
        this.paginator.pageIndex = 0;
        const filterArray: Array<string> = c.split('#');
        if (filterArray[0] === 'title') {
          this.riskSearchParameters.title = filterArray[1];
        } else if (filterArray[0] === 'createdFromDate') {
          if (filterArray[1] != 'null') {
            this.riskSearchParameters.createdFromDate = new Date(
              filterArray[1]
            );
            this.riskSearchParameters.createdToDate = this.createdToDateFilter;
          } else {
            this.riskSearchParameters.createdFromDate = undefined;
            this.riskSearchParameters.createdToDate = undefined;
          }
        } else if (filterArray[0] === 'createdToDate') {
          if (filterArray[1] != 'null') {
            this.riskSearchParameters.createdToDate = new Date(
              filterArray[1]
            );
            this.riskSearchParameters.createdFromDate =
              this.createdFromDateFilter;
          } else {
            this.riskSearchParameters.createdFromDate = undefined;
            this.riskSearchParameters.createdToDate = undefined;
          }
        } else if (filterArray[0] === 'createdToDate') {
          if (filterArray[1] != 'null') {
            this.riskSearchParameters.createdToDate = new Date(
              filterArray[1]
            );
            this.riskSearchParameters.createdFromDate =
              this.createdFromDateFilter;
          } else {
            this.riskSearchParameters.createdFromDate = undefined;
            this.riskSearchParameters.createdToDate = undefined;
          }
        } else if (filterArray[0] === 'createdToDate') {
          if (filterArray[1] != 'null') {
            this.riskSearchParameters.createdToDate = new Date(
              filterArray[1]
            );
            this.riskSearchParameters.createdFromDate =
              this.createdFromDateFilter;
          } else {
            this.riskSearchParameters.createdFromDate = undefined;
            this.riskSearchParameters.createdToDate = undefined;
          }
        } else if (filterArray[0] === 'riskCategoryId') {
          this.riskSearchParameters.riskCategoryId = filterArray[1];
        } else if (filterArray[0] === 'status') {
          this.riskSearchParameters.status = filterArray[1] ? parseInt(filterArray[1], 10) : undefined;
        } else if (filterArray[0] === 'riskScore') {
          const score = filterArray[1] ? parseInt(filterArray[1], 10) : undefined;
          if (score !== undefined && score > 25) {
            this.toastr.error('Risk score must not exceed 25');
            return;
          }
          this.riskSearchParameters.riskScore = score;
        } else if (filterArray[0] === 'riskLevel') {
          this.riskSearchParameters.riskLevel = filterArray[1];
        }
        this.refresh();
      });
  }

  clearOrderDates() {
    this._createdFromDateFilter = undefined;
    this._createdToDateFilter = undefined;
  }

  getRiskCategoryList() {
    this.sub$.sink = this.riskCategoryService.getRiskManagementCategory().subscribe({
      next: (data) => {
        const riskCategory = data as RiskManagementCategory[];
        if (riskCategory) {
          this.riskCateGoryList = [...riskCategory];
        }
        return;
      },
    });
  }

  refresh() {
    this.riskStore.loadByQuery(this.riskSearchParameters)
  }

  deleteRisk(riskId: string) {
    this.CommonDialogService
      .deleteConfirmtionDialog(
        this.translationService.getValue(
          'ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_RISK'
        )
      ).subscribe({
        next: (result: boolean) => {
          if (result) {
            this.riskStore.deleteRiskById(riskId);
          }
        },
      });
  }

  ngAfterViewInit(): void {
    this.sub$.sink = this.sort.sortChange.subscribe(() => {
      this.paginator.pageIndex = 0;
    });

    this.sub$.sink = merge(this.sort.sortChange, this.paginator.page)
      .pipe(
        tap(() => {
          this.riskSearchParameters.skip = this.paginator.pageIndex * this.paginator.pageSize;
          this.riskSearchParameters.pageSize = this.paginator.pageSize;
          this.riskSearchParameters.orderBy = this.sort.active + ' ' + this.sort.direction;
          this.refresh();
        })
      ).subscribe();
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.riskStore.riskList().indexOf(row);
  }
}
