import { HttpClient, HttpParams, HttpResponse } from '@angular/common/http';
import { inject, Injectable } from '@angular/core';
import { CommonHttpErrorService } from '@core/error-handler/common-http-error.service';
import { Risk } from '../model/risk';
import { catchError, Observable } from 'rxjs';
import { CommonError } from '@core/error-handler/common-error';
import { RiskResource } from '../model/risk-resource';

@Injectable({
  providedIn: 'root'
})
export class RiskService {
  private httpClient = inject(HttpClient);
  private commonHttpErrorService = inject(CommonHttpErrorService)
  constructor() { }

  getRisks(riskResource: RiskResource): Observable<HttpResponse<Risk[]>> {
    const url = 'Risk';
    const params = new HttpParams()
      .set('orderBy', riskResource.orderBy)
      .set('pageSize', riskResource.pageSize.toString())
      .set('skip', riskResource.skip.toString())
      .set('totalCount', riskResource.totalCount.toString())
      .set('title', riskResource.title || '')
      .set('assignedToId', riskResource.assignedToId || '')
      .set('riskCategoryId', riskResource.riskCategoryId || '')
      .set('createdFromDate', riskResource.createdFromDate ? riskResource.createdFromDate.toISOString() : '')
      .set('createdToDate', riskResource.createdToDate ? riskResource.createdToDate.toISOString() : '')
      .set('status', riskResource.status?.toString() || '')
      .set('riskScore', riskResource.riskScore ? riskResource.riskScore?.toString() : '')
      .set('riskLevel', riskResource.riskLevel || '');
    return this.httpClient
      .get<Risk[]>(url, {
        params: params,
        observe: 'response'
      });

  }

  createRisk(risk: Risk): Observable<Risk | CommonError> {
    const url = 'Risk';
    return this.httpClient
      .post<Risk>(url, risk)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  updateRisk(risk: Risk): Observable<Risk | CommonError> {
    const url = `Risk/${risk.id}`;
    return this.httpClient
      .put<Risk>(url, risk)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  deleteRisk(riskId: string): Observable<void | CommonError> {
    const url = `Risk/${riskId}`;
    return this.httpClient
      .delete<void>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getRiskById(riskId: string): Observable<Risk | CommonError> {
    const url = `Risk/${riskId}`;
    return this.httpClient
      .get<Risk>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getUserRiskControls(): Observable<Risk[] | CommonError> {
    const url = 'Risk/assigned';
    return this.httpClient
      .get<Risk[]>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError))
  }
}

