import { inject } from "@angular/core";
import { toObservable } from "@angular/core/rxjs-interop";
import { TranslationService } from "@core/services/translation.service";
import { tapResponse } from "@ngrx/operators";
import { signalStore, withState, withMethods, patchState, withHooks } from "@ngrx/signals";
import { rxMethod } from "@ngrx/signals/rxjs-interop";
import { ToastrService } from '@core/services/toastr-service';
import { pipe, tap, switchMap, distinctUntilChanged } from "rxjs";
import { Risk } from "../model/risk";
import { RiskService } from "./risk.service";
import { RiskResource } from "../model/risk-resource";
import { HttpResponse } from "@angular/common/http";



type RiskState = {
    riskList: Risk[];
    risk: Risk;
    loadList: boolean;
    isAddUpdate: boolean;
    filterParameters: RiskResource;
};

export const initialRiskState: RiskState = {
    riskList: [],
    risk: {} as Risk,
    loadList: true,
    isAddUpdate: false,
    filterParameters: {
        orderBy: 'title asc',
        pageSize: 10,
        skip: 0,
        totalCount: 0,
        title: '',
        assignedToId: '',
        riskCategoryId: '',
        createdToDate: null,
        createdFromDate: null,
        status: null,
        riskScore: null,
        riskLevel: '',
    }
};

export const RiskStore = signalStore(
    { providedIn: 'root' },
    withState(initialRiskState),
    withMethods(
        (
            store,
            riskService = inject(RiskService),
            toastrService = inject(ToastrService),
            translationService = inject(TranslationService),
        ) => ({
            loadByQuery: rxMethod<RiskResource>(
                pipe(
                    tap(() => patchState(store, { riskList: [] })),
                    switchMap((filter) => {
                        return riskService.getRisks(filter).pipe(
                            tapResponse({
                                next: (httpResponse: HttpResponse<Risk[]>) => {
                                    if (httpResponse && httpResponse.body) {
                                        patchState(store, { riskList: [...httpResponse.body], loadList: false })
                                    }
                                    if (httpResponse.headers.get('x-pagination')) {
                                        const pagination: RiskResource = JSON.parse(httpResponse.headers.get('x-pagination') ?? '');
                                        patchState(store, { filterParameters: { ...filter, totalCount: pagination.totalCount } })
                                    }
                                },
                                error: (error: any) => {
                                    toastrService.error(error.error);
                                }
                            })
                        );
                    })
                )
            ),
            addRisk: rxMethod<Risk>(
                pipe(
                    distinctUntilChanged(),
                    tap(() => patchState(store, { loadList: false })),
                    switchMap((risk: Risk) =>
                        riskService.createRisk(risk).pipe(
                            tapResponse({
                                next: () => {
                                    toastrService.success(
                                        translationService.getValue('RISK_CREATED_SUCCESSFULLY')
                                    );
                                    patchState(store, { isAddUpdate: true, loadList: true });
                                },
                                error: (err: any) => {
                                    console.error(err);
                                },
                            })
                        )
                    )
                )
            ),
            updateRisk: rxMethod<Risk>(
                pipe(
                    distinctUntilChanged(),
                    tap(() => patchState(store, { loadList: false })),
                    switchMap((risk: Risk) =>
                        riskService.updateRisk(risk).pipe(
                            tapResponse({
                                next: (response) => {
                                    const updatedRisk: Risk = response as Risk;
                                    toastrService.success(
                                        translationService.getValue('RISK_UPDATED_SUCCESSFULLY')
                                    );
                                    patchState(store, {
                                        riskList: store.riskList().map((cr) => {
                                            return updatedRisk.id === cr.id ? updatedRisk : cr;
                                        }),
                                        isAddUpdate: true,
                                    });
                                },
                                error: (err: any) => {
                                    console.error(err);
                                },
                            })
                        )
                    )
                )
            ),
            deleteRiskById: rxMethod<string>(
                pipe(
                    distinctUntilChanged(),
                    tap(() => patchState(store, { loadList: false })),
                    switchMap((riskId: string) =>
                        riskService.deleteRisk(riskId).pipe(
                            tapResponse({
                                next: () => {
                                    toastrService.success(
                                        translationService.getValue('RISK_DELETED_SUCCESSFULLY')
                                    );
                                    patchState(store, {
                                        riskList: store.riskList().filter((w) => w.id !== riskId),
                                        filterParameters: {
                                            ...store.filterParameters(),
                                            totalCount: store.filterParameters().totalCount - 1,
                                        },
                                    });
                                },
                                error: (err: any) => {
                                    console.error(err);
                                },
                            })
                        )
                    )
                )
            ),
            resetflag() {
                patchState(store, { isAddUpdate: false })
            },
        })
    ),
    withHooks({
        onInit(store) {
            toObservable(store.loadList).subscribe((flag) => {
                if (flag) {
                    store.loadByQuery(store.filterParameters());
                }
            });
        },
    })
);
