import { Component, inject, OnInit } from '@angular/core';
import { BaseComponent } from '../../base.component';
import { RoleService } from '../role.service';
import { Role } from '@core/domain-classes/role';
import { ActivatedRoute, Router, RouterLink } from '@angular/router';
import { ToastrService } from '@core/services/toastr-service';
import { FormsModule } from '@angular/forms';
import { MatButtonModule } from '@angular/material/button';
import { MatCardModule } from '@angular/material/card';
import { MatCheckboxChange, MatCheckboxModule } from '@angular/material/checkbox';
import { MatExpansionModule } from '@angular/material/expansion';
import { MatIconModule } from '@angular/material/icon';
import { TranslateModule } from '@ngx-translate/core';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { CommonService } from '@core/services/common.service';
import { Module } from '@core/domain-classes/module';
import { Page } from '@core/domain-classes/page';
import { Action } from '@core/domain-classes/action';

@Component({
  selector: 'app-manage-role',
  templateUrl: './manage-role.component.html',
  styleUrls: ['./manage-role.component.scss'],
  standalone: true,
  imports: [
    PageHelpTextComponent,
    FormsModule,
    MatCheckboxModule,
    RouterLink,
    TranslateModule,
    MatIconModule,
    MatCardModule,
    MatButtonModule,
    MatExpansionModule
  ]
})
export class ManageRoleComponent extends BaseComponent implements OnInit {
  role: Role = { roleClaims: [], userRoles: [] };
  commonService = inject(CommonService);
  modules: Module[] = [];

  constructor(
    private activeRoute: ActivatedRoute,
    private router: Router,
    private toastrService: ToastrService,
    private roleService: RoleService) {
    super();
  }

  ngOnInit(): void {
    this.loadPermissions();

    this.sub$.sink = this.activeRoute.data.subscribe(
      (data: any) => {
        if (data.role) {
          this.role = data.role;
        } else {
          this.role = {
            roleClaims: [],
            userRoles: []
          };
        }
      });
  }

  loadPermissions(): void {
    this.sub$.sink = this.commonService
      .getAllPermissions()
      .subscribe((permissions) => {
        this.modules = permissions;
      });
  }

  manageRole(role: Role): void {
    if (!role.name) {
      this.toastrService.error(this.translationService.getValue('PLEASE_ENTER_ROLE_NAME'));
      return;
    }

    if (role.roleClaims?.length == 0) {
      this.toastrService.error(this.translationService.getValue('PLEASE_SELECT_AT_LEAT_ONE_PERMISSION'));
      return;
    }

    if (!role.id)
      this.sub$.sink = this.roleService.addRole(role).subscribe(() => {
        this.toastrService.success(this.translationService.getValue('ROLE_SAVED_SUCCESSFULLY'));
        this.router.navigate(['/roles']);
      });
    else
      this.sub$.sink = this.roleService.updateRole(role).subscribe(() => {
        this.toastrService.success(this.translationService.getValue('ROLE_UPDATED_SUCCESSFULLY'));
        this.router.navigate(['/roles']);
      });
  }

  onPageSelect(event: MatCheckboxChange, page: Page) {
    if (event.checked) {
      page.actions.forEach((action) => {
        if (!this.checkPermission(action.id)) {
          this.role.roleClaims?.push({
            roleId: this.role.id,
            claimType: action.code,
            claimValue: '',
            actionId: action.id,
          });
        }
      });
    } else {
      var actions = page.actions?.map((c) => c.id);
      this.role.roleClaims = this.role.roleClaims?.filter(
        (c) => actions.indexOf(c.actionId) < 0
      );
    }
  }

  selecetAll(event: MatCheckboxChange) {
    if (event.checked) {
      this.modules.forEach((module) => {
        module.pages.forEach((page) => {
          page.actions.forEach((action) => {
            if (!this.checkPermission(action.id)) {
              this.role.roleClaims?.push({
                roleId: this.role.id,
                claimType: action.code,
                claimValue: '',
                actionId: action.id,
              });
            }
          });
        });
      });
    } else {
      this.role.roleClaims = [];
    }
  }

  checkPermission(actionId: string): boolean {
    const pageAction = this.role.roleClaims?.find(
      (c) => c.actionId === actionId
    );
    if (pageAction) {
      return true;
    } else {
      return false;
    }
  }

  onPermissionChange(flag: any, page: Page, action: Action) {
    if (flag.checked) {
      this.role.roleClaims?.push({
        roleId: this.role.id,
        claimType: action.code,
        claimValue: '',
        actionId: action.id,
      });
    } else {
      const roleClaimToRemove = this.role.roleClaims?.find(
        (c) => c.actionId === action.id
      );
      if (roleClaimToRemove) {
        const index = this.role.roleClaims?.indexOf(roleClaimToRemove, 0);
        if (index !== undefined && index > -1) {
          this.role.roleClaims?.splice(index, 1);
        }
      }
    }
  }

  isModuleFullySelected(module: Module): boolean {
    if (!module.pages?.length) return false;
    for (const page of module.pages) {
      if (!page.actions?.length) continue;
      for (const action of page.actions) {
        if (!this.checkPermission(action.id)) {
          return false;
        }
      }
    }
    return true;
  }

  isModuleIndeterminate(module: Module): boolean {
    let total = 0;
    let selected = 0;
    if (!module.pages?.length) return false;
    for (const page of module.pages) {
      if (!page.actions?.length) continue;
      for (const action of page.actions) {
        total++;
        if (this.checkPermission(action.id)) {
          selected++;
        }
      }
    }
    return selected > 0 && selected < total;
  }

  onModuleSelect(event: MatCheckboxChange, module: Module) {
    if (event.checked) {
      module.pages?.forEach(page => {
        page.actions?.forEach(action => {
          if (!this.checkPermission(action.id)) {
            this.role.roleClaims?.push({
              roleId: this.role.id,
              claimType: action.code,
              claimValue: '',
              actionId: action.id,
            });
          }
        });
      });
    } else {
      // Remove all actions in this module from roleClaims
      const actionIds: string[] = [];
      module.pages?.forEach(page => {
        page.actions?.forEach(action => {
          actionIds.push(action.id);
        });
      });
      this.role.roleClaims = this.role.roleClaims?.filter(rc => !actionIds.includes(rc.actionId));
    }
  }

  isPageFullySelected(page: Page): boolean {
    if (!page.actions?.length) return false;
    return page.actions.every(action => this.checkPermission(action.id));
  }

  isPageIndeterminate(page: Page): boolean {
    if (!page.actions?.length) return false;
    const selected = page.actions.filter(action => this.checkPermission(action.id)).length;
    return selected > 0 && selected < page.actions.length;
  }
}
