import { HttpEvent, HttpEventType } from '@angular/common/http';
import { Component, inject, Input, OnChanges } from '@angular/core';
import {  ProgressSpinnerMode } from '@angular/material/progress-spinner';
import { AllowFileExtension } from '@core/domain-classes/allow-file-extension';
import { CourseSessionMediaChunk } from '@core/domain-classes/course-session-media-chunk';
import { CourseSessionMediaChunkDownload } from '@core/domain-classes/course-session-media-chunk-download';
import { DocumentChunkDownload } from '@core/domain-classes/document-chunk-download';
import { EmployeeCourseSessionActivity } from '@core/domain-classes/employee-course-session-activity';
import { EmployeeCourseSessionActivityType } from '@core/domain-classes/employee-course-session-activity.enum';
import { FileType } from '@core/domain-classes/file-type.enum';
import { MediaPreview } from '@core/domain-classes/media-previw-data';
import { MediaType } from '@core/domain-classes/session-media-type-enum';
import { CommonError } from '@core/error-handler/common-error';
import { ClonerService } from '@core/services/clone.service';
import { CommonService } from '@core/services/common.service';
import { AudioPreviewComponent } from '@shared/audio-preview/audio-preview.component';
import { CSVPreviewComponent } from '@shared/csv-preview/csv-preview.component';
import { ImagePreviewComponent } from '@shared/image-preview/image-preview.component';
import { JsonPreviewComponent } from '@shared/json-preview/json-preview.component';
import { NoPreviewComponent } from '@shared/no-preview/no-preview.component';
import { OfficeViewerComponent } from '@shared/office-viewer/office-viewer.component';
import { OverlayPanel } from '@shared/overlay-panel/overlay-panel.service';
import { PdfDataViewerComponent } from '@shared/pdf-viewer/pdf-viewer.component';
import { TextPreviewComponent } from '@shared/text-preview/text-preview.component';
import { VideoPreviewComponent } from '@shared/video-preview/video-preview.component';
import { VimeoVideoViewerComponent } from '@shared/vimeo-video-viewer/vimeo-video-viewer.component';
import { YoutubeVideoViewerComponent } from '@shared/youtube-video-viewer/youtube-video-viewer.component';

import { ToastrService } from '@core/services/toastr-service';
import { bufferCount, concatMap, from, mergeMap } from 'rxjs';
import { BaseComponent } from '../../base.component';
import { SessionMediaService } from '../../training/course-session/session-media/session-meida.service';

@Component({
  selector: 'app-base-preview-media',
  imports: [
    ImagePreviewComponent,
    CSVPreviewComponent,
    OfficeViewerComponent,
    JsonPreviewComponent,
    PdfDataViewerComponent,
    VideoPreviewComponent,
    TextPreviewComponent,
    AudioPreviewComponent,
    YoutubeVideoViewerComponent,
    VimeoVideoViewerComponent,
    NoPreviewComponent
  ],
  templateUrl: './base-preview-media.component.html',
  styleUrl: './base-preview-media.component.scss'
})
export class BasePreviewMediaComponent extends BaseComponent implements OnChanges {
  @Input() preview: Partial<MediaPreview>;
  type = '';
  currentDoc: MediaPreview;
  allowFileExtension: AllowFileExtension[] = [];
  isDownloadFlag: boolean = false;
  isDocumentChange: boolean = false;
  contentType: string = '';
  fileType = FileType;
  mode: ProgressSpinnerMode = 'determinate';
  downloadCountPercentage = 0;
  sessionMediaService = inject(SessionMediaService);
  employeeCourseSessionActivity?: EmployeeCourseSessionActivity;
  documentChunkDownloads: DocumentChunkDownload[] = [];
  documentChunks: CourseSessionMediaChunk[] = [];
  documentUrl: Blob;
  isChunk: boolean = false;

  constructor(
    public overlay: OverlayPanel,
    private commonService: CommonService,
    private clonerService: ClonerService,
    private toastrService: ToastrService,
  ) {
    super();
  }

  ngOnChanges() {
    this.getAllowExtension();
  }

  closeToolbar() {
    this.overlay.setIsClosePanelClose(this.isDocumentChange);
  }

  onDocumentChange(flag: boolean) {
    this.isDocumentChange = flag;
    this.overlay.setIsClosePanelClose(this.isDocumentChange);
  }

  getAllowExtension() {
    this.sub$.sink = this.commonService.allowFileExtension$.subscribe(
      (allowFileExtension: AllowFileExtension[]) => {
        this.allowFileExtension = allowFileExtension;
        this.onDocumentView(this.preview);
      }
    );
  }

  onDocumentView(document: Partial<MediaPreview>) {
    this.currentDoc = this.clonerService.deepClone<MediaPreview>(document);
    if (document.isTracking) {
      this.addDocumentTrail(document);
    }
    if (this.currentDoc.mediaType == MediaType.FILE) {
      const extension = document.extension?.split('.').pop();
      const allowTypeExtenstion = this.allowFileExtension.find((c) =>
        c.extensions?.find(
          (ext: string) => ext.toLowerCase() === document.extension?.toLowerCase() || ext.toLowerCase() === extension?.toLowerCase()
        )
      );
      this.type = this.fileType[allowTypeExtenstion?.fileType ?? this.fileType.Other]?.toLowerCase();
      let isLive = true;
      if (window.location.hostname === "localhost" || window.location.hostname === "127.0.0.1") {
        isLive = false;
      }

      if (!(this.type?.toLowerCase() == 'office' || this.type?.toLowerCase() == 'text') || (!isLive && (this.currentDoc.extension === 'xlsx' || this.currentDoc.extension === 'xls')) || (!isLive && (this.currentDoc.extension === 'doc' || this.currentDoc.extension === 'docx'))) {
        {
          this.checkDocumentChunk();
        }
      }
    } else if (this.currentDoc.mediaType == MediaType.YOUTUBE_VIDEO_LINK) {
      this.type = 'youtube';
    } else if (this.currentDoc.mediaType == MediaType.VIMEO_VIDEO_LINK) {
      this.type = 'vimeo';
    } else if (this.currentDoc.mediaType == MediaType.EXTERNAL_LINK) {
      window.open(this.currentDoc.url, '_blank');
    }
  }

  addDocumentTrail(media: Partial<MediaPreview>) {
    this.employeeCourseSessionActivity = {
      courseSessionMediaId: media.id ?? '',
      activityType: EmployeeCourseSessionActivityType.Media_Viewed,
      startDate: new Date().toISOString(),
      link: media.employeeCourseSessionLink,
    };
  }

  checkDocumentChunk() {
    const id = this.preview ? this.preview.id : '';
    this.sub$.sink = this.sessionMediaService
      .checkDocumentStoreAsChunk(id ?? '')
      .subscribe((c) => {
        this.isChunk = c;
        this.getDocument();
      });
  }

  getDocument() {
    if (this.isChunk) {
      this.getAllDocumentChunks();
    } else {
      this.mode = 'indeterminate';
      this.sub$.sink = this.sessionMediaService
        .downloadDocument(this.currentDoc)
        .subscribe({
          next: (response: HttpEvent<Blob> | CommonError) => {
            const event = response as HttpEvent<Blob>;
            if (event && event.type === HttpEventType.Response) {
              const blobParth = event?.body as BlobPart;
              this.documentUrl = new Blob([blobParth], { type: event?.body?.type });
            }
          },
          error: (err: any) => {
            this.toastrService.error(err.error.message);
          },
        });
    }

  }

  getAllDocumentChunks() {
    const documentId = this.currentDoc?.id;
    this.sub$.sink = this.sessionMediaService.getDocumentChunks(documentId ?? '').subscribe({
      next: (data) => {
        this.documentChunks = data;
        if (this.documentChunks.length > 0) {
          this.startDownload();
        }
      },
      error: (err: any) => {
        this.toastrService.error(err.error.message);
      },
    });
  }

  startDownload() {
    const { chunkSize1, parallelCalls } = this.commonService.getNetworkSpeed();
    let completedChunks = 0;
    const chunkRequests = [];
    for (let i = 0; i < this.documentChunks.length; i++) {
      chunkRequests.push({ chunkIndex: i, sessionMediaId: this.documentChunks[i].courseSessionMediaId });
    }
    from(chunkRequests)
      .pipe(
        bufferCount(parallelCalls), // Group requests into batches of 5
        concatMap((batch) =>
          from(batch).pipe(
            mergeMap((chunk) => this.downloadChunk(chunk.chunkIndex, chunk.sessionMediaId), parallelCalls) // Retrieve 5 chunks in parallel
          )
        ),
      )
      .subscribe({
        next: (response) => {
          const documentChunkDownload = response as CourseSessionMediaChunkDownload;
          completedChunks++;
          this.downloadCountPercentage = Math.round((completedChunks / chunkRequests.length) * 100);
          this.contentType = documentChunkDownload.contentType;
          const chunkBlob = this.base64ToBlob(documentChunkDownload.data, documentChunkDownload.contentType);
          documentChunkDownload.blobChunk = chunkBlob;
          this.documentChunkDownloads.push(documentChunkDownload);
        },
        complete: () => this.mergeChunks()
      });
  }

  private base64ToBlob(base64: string, contentType: string): Blob {
    const byteCharacters = atob(base64);
    const byteNumbers = new Array(byteCharacters.length);
    for (let i = 0; i < byteCharacters.length; i++) {
      byteNumbers[i] = byteCharacters.charCodeAt(i);
    }
    const byteArray = new Uint8Array(byteNumbers);
    return new Blob([byteArray], { type: contentType });
  }

  downloadChunk(chunkIndex: number, documentVersionId: string) {
    return this.sessionMediaService.downloadDocumentChunk(documentVersionId, chunkIndex)
  }

  mergeChunks() {
    this.downloadCountPercentage = 100;
    const sortedChunks = this.documentChunkDownloads
      .sort((a, b) => a.chunkIndex - b.chunkIndex)
      .map(entry => entry.blobChunk)
      .filter((chunk): chunk is Blob => chunk !== undefined);
    const blob = new Blob(sortedChunks, { type: this.contentType });
    this.documentUrl = blob;
  }
}
