import { Injectable } from '@angular/core';
import { BehaviorSubject, Observable, throwError } from 'rxjs';
import { Router, NavigationStart, NavigationEnd, NavigationCancel, NavigationError } from '@angular/router';
import { HttpInterceptor, HttpRequest, HttpHandler, HttpEvent } from '@angular/common/http';
import { finalize, catchError } from 'rxjs/operators';

@Injectable({ providedIn: 'root' })
export class PendingInterceptorService implements HttpInterceptor {
  private _pendingRequests = 0;
  private _pendingRequestsStatus = new BehaviorSubject<boolean>(false);

  get pendingRequestsStatus(): Observable<boolean> {
    return this._pendingRequestsStatus.asObservable();
  }

  private _filteredUrlPatterns: RegExp[] = [];

  constructor(router: Router) {
    router.events.subscribe(event => {
      if (event instanceof NavigationStart) {
        this._pendingRequestsStatus.next(true);
      }

      if (event instanceof NavigationEnd || event instanceof NavigationCancel || event instanceof NavigationError) {
        this._pendingRequestsStatus.next(false);
      }
    });
  }

  intercept(req: HttpRequest<any>, next: HttpHandler): Observable<HttpEvent<any>> {
    const shouldBypass = this._filteredUrlPatterns.some(e => e.test(req.url));

    if (!shouldBypass) {
      this._pendingRequests++;
      if (this._pendingRequests === 1) this._pendingRequestsStatus.next(true);
    }

    return next.handle(req).pipe(
      catchError(err => throwError(() => err)),
      finalize(() => {
        if (!shouldBypass) {
          this._pendingRequests--;

          if (this._pendingRequests <= 0) {
            this._pendingRequests = 0;
            this._pendingRequestsStatus.next(false);
          }
        }
      })
    );
  }
}
