import { NgClass } from '@angular/common';
import { Component, inject, OnInit } from '@angular/core';
import {
  FormBuilder,
  FormGroup,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { MatCardModule } from '@angular/material/card';
import { MatIconModule } from '@angular/material/icon';
import { MatSelectModule } from '@angular/material/select';
import { ActivatedRoute, Router, RouterModule } from '@angular/router';
import { Supplier } from '../../supplier/model/supplier';
import { TranslateModule } from '@ngx-translate/core';
import { BaseComponent } from '../../base.component';
import { SupplierStore } from '../supplier-store';
import { SupplierStatus } from '../../supplier/model/supplier-status';
import { SupplierService } from '../supplier.service';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { MatButtonModule } from '@angular/material/button';
import { toObservable } from '@angular/core/rxjs-interop';

@Component({
  selector: 'app-manage-supplier',
  imports: [
    TranslateModule,
    NgClass,
    RouterModule,
    ReactiveFormsModule,
    MatSelectModule,
    MatIconModule,
    MatCardModule,
    PageHelpTextComponent,
    MatButtonModule
  ],
  templateUrl: './manage-supplier.component.html',
  styleUrl: './manage-supplier.component.scss',
})
export class ManageSupplierComponent extends BaseComponent implements OnInit {
  supplier: Supplier;
  supplierForm: FormGroup;
  isDetailsMode = true;
  isEditMode = false;
  fb = inject(FormBuilder);
  router = inject(Router);
  activeRoute = inject(ActivatedRoute);
  supplierService = inject(SupplierService);
  supplierStore = inject(SupplierStore);

  statusList = Object.keys(SupplierStatus)
    .filter((key) => !isNaN(Number(SupplierStatus[key as any])))
    .map((key) => ({
      label: key,
      value: SupplierStatus[key as keyof typeof SupplierStatus],
    }));

  constructor() {
    super();
    this.subscribeIsAddorUpdate();
  }

  ngOnInit(): void {
    this.createSupplierForm();
    this.sub$.sink = this.activeRoute.data.subscribe(
      (data) => {
        const supplierData = data as { supplier: Supplier };
        if (supplierData.supplier) {
          this.isEditMode = true;
          this.supplierForm.patchValue(supplierData.supplier);
          this.supplier = supplierData.supplier;
        } else {
          this.isEditMode = false;
          this.getSupplierCode();
        }
      }
    );
    this.sub$.sink = this.activeRoute.queryParams.subscribe((params) => {
      if (params['isDetailsMode'] === 'true') {
        this.isDetailsMode = false;
      } else {
        this.supplierForm.enable();
      }
    });
  }

  getSupplierCode() {
    this.sub$.sink = this.supplierService.getSupplierCode().subscribe((sc) => {
      const supplier = sc as Supplier;
      if (supplier) {
        this.supplierForm.patchValue({
          supplierCode: supplier.supplierCode,
        });
      }
    });
  }

  createSupplierForm() {
    this.supplierForm = this.fb.group(
      {
        id: [''],
        supplierCode: ['', [Validators.required]],
        name: ['', [Validators.required]],
        contactPerson: ['', [Validators.required]],
        email: ['', [Validators.email]],
        phone: [''],
        status: ['', [Validators.required]],
        country: [''],
        address: [''],
      },
      {
        validators: [this.emailOrPhoneRequiredValidator],
      }
    );
  }

  emailOrPhoneRequiredValidator(formGroup: FormGroup) {
    const email = formGroup.get('email')?.value;
    const phone = formGroup.get('phone')?.value;

    if (!email && !phone) {
      return { emailOrPhoneRequired: true };
    }
    return null;
  }

  saveSupplier() {
    if (this.supplierForm.invalid) {
      this.supplierForm.markAllAsTouched();
      return;
    }

    const supplier = this.supplierForm.getRawValue() as Supplier;
    if (this.isEditMode) {
      this.supplierStore.updateSupplier(supplier);
    } else {
      this.supplierStore.addSupplier(supplier);
    }
  }

  subscribeIsAddorUpdate() {
    toObservable(this.supplierStore.isAddUpdate).subscribe((flag) => {
      if (flag) {
        this.router.navigate(['/supplier/list']);
      }
      this.supplierStore.resetflag();
    });
  }
}
