import { Component, Inject, inject, OnInit, Optional } from '@angular/core';
import {
  FormBuilder,
  FormGroup,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { MatButtonModule } from '@angular/material/button';
import { MatCardModule } from '@angular/material/card';
import { MatSelectModule } from '@angular/material/select';
import { ActivatedRoute, RouterLink } from '@angular/router';
import { SecurityService } from '@core/security/security.service';

import { ToastrService } from '@core/services/toastr-service';
import { BaseComponent } from '../../../base.component';
import { UserStore } from '../../../user/store/user.store';
import { SupplierAuditType } from '../../model/supplier-audit-type.enum';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { SupplierResult } from '../../model/supplier-result.enum';
import { SupplierAudit } from '../../model/supplier-audit';
import { SupplierAuditService } from '../supplier-audit.service';
import { SupplierAuditListComponent } from '../supplier-audit-list/supplier-audit-list.component';
import { MatDialogRef, MAT_DIALOG_DATA } from '@angular/material/dialog';
import { NgClass } from '@angular/common';
import { MatIconModule } from '@angular/material/icon';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';

@Component({
  selector: 'app-manage-supplier-audit',
  imports: [
    ReactiveFormsModule,
    MatSelectModule,
    MatButtonModule,
    MatCardModule,
    MatIconModule,
    SupplierAuditListComponent,
    RouterLink,
    MatDatepickerModule,
    NgClass,
    HasClaimDirective,
    TranslateModule
  ],
  templateUrl: './manage-supplier-audit.component.html',
  styleUrl: './manage-supplier-audit.component.scss',
})
export class ManageSupplierAuditComponent
  extends BaseComponent
  implements OnInit {
  supplierAuditForm: FormGroup;
  fb = inject(FormBuilder);
  supplierAuditService = inject(SupplierAuditService);
  userStore = inject(UserStore);
  securityService = inject(SecurityService);
  toastrService = inject(ToastrService);
  router = inject(ActivatedRoute);
  supplierAuditResponse: SupplierAudit = {} as SupplierAudit;
  minDueDate: Date = new Date();
  auditTypeList = Object.keys(SupplierAuditType)
    .filter((key) => !isNaN(Number(SupplierAuditType[key as any])))
    .map((key) => ({
      label: key,
      value: SupplierAuditType[key as keyof typeof SupplierAuditType],
    }));

  supplierResultList = Object.keys(SupplierResult)
    .filter((key) => isNaN(Number(key)))
    .map((key) => ({
      label: key,
      value: SupplierResult[key as keyof typeof SupplierResult],
    }));

  constructor(
    @Optional() private dialogRef: MatDialogRef<ManageSupplierAuditComponent>,
    @Optional() @Inject(MAT_DIALOG_DATA) public data: { supplierAudit: SupplierAudit, isViewOnly: boolean },
  ) {
    super();
  }

  ngOnInit(): void {
    this.creatsupplierAuditForm();

    this.router.parent?.paramMap.subscribe((params) => {
      const supplierId = params.get('id');
      if (supplierId) {
        this.supplierAuditForm.patchValue({ supplierId: supplierId });
      }
    });

    this.supplierAuditForm.get('result')?.valueChanges.subscribe((val) => {
      const otherResultCtrl = this.supplierAuditForm.get('otherResult');

      if (val === 'Conditional') {
        otherResultCtrl?.setValidators([Validators.required]);
      } else {
        otherResultCtrl?.clearValidators();
      }

      otherResultCtrl?.updateValueAndValidity();
    });

    if (this.data !== null) {
      this.supplierAuditForm.patchValue(this.data.supplierAudit);
    } else {
      this.data = {
        supplierAudit: {} as SupplierAudit,
        isViewOnly: false,
      };
    }
  }

  creatsupplierAuditForm(): void {
    this.supplierAuditForm = this.fb.group({
      id: [''],
      supplierId: [''],
      auditorId: [
        this.securityService.securityObject.id,
        [Validators.required],
      ],
      auditType: ['', [Validators.required]],
      auditDate: [new Date, [Validators.required]],
      findings: ['', [Validators.required]],
      score: [
        '',
        [Validators.required, Validators.min(0), Validators.max(100)],
      ],
      result: ['', [Validators.required]],
      otherResult: [''],
    });
  }

  onSubmit(): void {
    const form = this.supplierAuditForm;
    const resultCtrl = form.get('result');
    const otherResultCtrl = form.get('otherResult');

    if (resultCtrl?.value === 'Conditional' && otherResultCtrl?.value?.trim()) {
      resultCtrl.setValue(otherResultCtrl.value.trim(), { emitEvent: false });
    }

    if (form.invalid) {
      form.markAllAsTouched();
      return;
    }

    const supplierAudit: SupplierAudit = form.getRawValue();

    if (!supplierAudit.supplierId) return;

    this.supplierAuditService.addSupplierAudit(supplierAudit).subscribe({
      next: (response: SupplierAudit) => {
        this.toastrService.success(
          this.translationService.getValue('SUPPLIER_AUDIT_CREATE_SUCCESSFULLY')
        );
        this.supplierAuditResponse = response;
        form.reset();
        form.patchValue({
          supplierId: response.supplierId,
          auditorId: this.securityService.securityObject.id,
          auditDate: new Date()
        });
      },
    });
  }

  onCancel(): void {
    if (this.dialogRef) {
      this.dialogRef.close();
    }
  }
}
