import {
  Component,
  inject,
  Input,
  OnChanges,
  OnInit,
  SimpleChanges,
} from '@angular/core';
import { MatTableModule } from '@angular/material/table';
import { BaseComponent } from '../../../base.component';
import { SupplierAudit } from '../../model/supplier-audit';
import { SupplierAuditService } from '../supplier-audit.service';
import { MatDialog } from '@angular/material/dialog';
import { ManageSupplierAuditComponent } from '../manage-supplier-audit/manage-supplier-audit.component';
import { TranslateModule } from '@ngx-translate/core';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { ActivatedRoute } from '@angular/router';
import { SupplierAuditTypePipe } from '../../pipes/supplier-audit-type.pipe';
import { NgClass } from '@angular/common';
import { SupplierAuditResultPipe } from '../../pipes/supplier-audit-result.pipe';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';
import { MatButtonModule } from '@angular/material/button';

@Component({
  selector: 'app-supplier-audit-list',
  imports: [
    MatTableModule,
    TranslateModule,
    UTCToLocalTime,
    SupplierAuditTypePipe,
    SupplierAuditResultPipe,
    NgClass,
    MatIconModule,
    MatCardModule,
    MatButtonModule
  ],
  templateUrl: './supplier-audit-list.component.html',
  styleUrl: './supplier-audit-list.component.scss',
})
export class SupplierAuditListComponent
  extends BaseComponent
  implements OnInit, OnChanges {
  @Input() supplierAuditResponse: SupplierAudit | null = null;
  supplierAuditList: SupplierAudit[] = [];
  displayedColumns: string[] = ['action', 'auditor', 'auditType', 'auditDate', 'score', 'result'];
  supplierAuditService = inject(SupplierAuditService);
  dialog = inject(MatDialog);
  route = inject(ActivatedRoute);

  ngOnInit(): void {
    this.route.parent?.params.subscribe(params => {
      const supplierId = params['id'];
      this.getAllSupplierAudits(supplierId);
    });
  }

  ngOnChanges(changes: SimpleChanges): void {
    if (changes['supplierAuditResponse'] && this.supplierAuditResponse) {
      this.supplierAuditList = [...this.supplierAuditList, this.supplierAuditResponse];
    }
  }

  viewSupplierAudit(supplierAudit: SupplierAudit) {
    this.dialog.open(ManageSupplierAuditComponent, {
      maxWidth: '60vw',
      width: '100%',
      data: { supplierAudit, isViewOnly: true },
    });
  }

  getAllSupplierAudits(supplierId: string) {
    this.sub$.sink = this.supplierAuditService
      .getSupplierAudit(supplierId)
      .subscribe({
        next: (reviews: SupplierAudit[]) => {
          const supplierAudits = reviews;
          if (supplierAudits) {
            this.supplierAuditList = supplierAudits;
          } else {
            this.supplierAuditList = [];
          }
        },
        error: (error) => {
          console.error('Error fetching reviews:', error);
        },
      });
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.supplierAuditList.indexOf(row);
  }
}
