import { ChangeDetectorRef, Component, inject, OnInit } from '@angular/core';
import { FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { MatCardModule } from '@angular/material/card';
import { MatSelectModule } from '@angular/material/select';
import { SupplierCertificationListComponent } from '../supplier-certification-list/supplier-certification-list.component';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { BaseComponent } from '../../../base.component';
import { SupplierCertification } from '../../model/supplier-certification';
import { SUPPLIER_CERTIFICATION_STATUS } from '../../model/supplier-certification-status.enum';
import { MatButtonModule } from '@angular/material/button';
import { ActivatedRoute, RouterLink } from '@angular/router';
import { MatIconModule } from '@angular/material/icon';
import { AllowFileExtension } from '@core/domain-classes/allow-file-extension';
import { CommonService } from '@core/services/common.service';

import { ToastrService } from '@core/services/toastr-service';
import { validateFile } from '@core/domain-classes/extension-types';
import { SupplierCertificationService } from '../supplier-certification.service';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';

@Component({
  selector: 'app-manage-supplier-certification',
  imports: [
    ReactiveFormsModule,
    MatSelectModule,
    MatCardModule,
    SupplierCertificationListComponent,
    MatDatepickerModule,
    MatButtonModule,
    RouterLink,
    MatIconModule,
    HasClaimDirective,
    TranslateModule
  ],
  templateUrl: './manage-supplier-certification.component.html',
  styleUrl: './manage-supplier-certification.component.scss'
})
export class ManageSupplierCertificationComponent extends BaseComponent implements OnInit {
  supplierCertificationForm: FormGroup;
  fb = inject(FormBuilder);
  minQualificationDate: Date = new Date();
  supplierCertification: SupplierCertification = {} as SupplierCertification;
  isDragging = false;
  allowFileExtension: AllowFileExtension[] = [];
  commonService = inject(CommonService);
  toastrService = inject(ToastrService);
  cd = inject(ChangeDetectorRef);
  supplierCertificationService = inject(SupplierCertificationService);
  route = inject(ActivatedRoute);

  supplierStatusList = Object.keys(SUPPLIER_CERTIFICATION_STATUS)
    .filter((key) => isNaN(Number(key)))
    .map((key) => ({
      label: key,
      value: SUPPLIER_CERTIFICATION_STATUS[key as keyof typeof SUPPLIER_CERTIFICATION_STATUS],
    }));

  ngOnInit(): void {
    this.createSupplierCertificationForm();
    this.getAllAllowFileExtension();

    this.route.parent?.params.subscribe((params) => {
      const supplierId = params['id'];
      if (supplierId) {
        this.supplierCertificationForm.get('supplierId')?.setValue(supplierId);
      }
    });
  }

  createSupplierCertificationForm(): void {
    this.supplierCertificationForm = this.fb.group({
      id: [''],
      supplierId: [''],
      certificationName: ['', [Validators.required]],
      issuedBy: ['', [Validators.required]],
      issuedDate: [null, [Validators.required]],
      expiryDate: [null],
      certificationNumber: ['', [Validators.required]],
      status: ['', [Validators.required]],
      filePath: [''],
      fileName: ['', [Validators.required]],
      file: [null, [Validators.required]],
    });
  }

  getAllAllowFileExtension() {
    this.sub$.sink = this.commonService.allowFileExtension$.subscribe(
      (allowFileExtension: AllowFileExtension[]) => {
        if (allowFileExtension) {
          this.allowFileExtension = allowFileExtension;
        }
      }
    );
  }

  async upload(files: FileList | null) {
    if (files == null) return;
    if (files.length === 0) return;

    const extension = files[0].name.split('.').pop() ?? '';
    this.supplierCertificationForm?.get('file')?.setValue(files[0]);
    this.supplierCertificationForm?.get('fileName')?.setValue(files[0].name);
    await this.checkDocument(files[0], extension);
  }

  async checkDocument(files: File, extension: string) {
    if (!(await validateFile(files))) {
      this.toastrService.error(this.translationService.getValue('INVALID_EXTENSION_OR_CORRUPT_INVALID_SIGNATURE'));
      this.cd.markForCheck();
      return;
    } else if (!this.fileExtensionValidation(extension)) {
      this.toastrService.error(this.translationService.getValue('PLEASE_SELECT_VALID_FILE_EXTENSION'));
      return;
    }
  }

  fileExtensionValidation(extension: string): boolean {
    const allowTypeExtension = this.allowFileExtension.find((c) =>
      c.extensions?.find((ext) => ext.toLowerCase() === extension.toLowerCase())
    );
    return allowTypeExtension ? true : false;
  }

  onSubmit(file: HTMLInputElement): void {
    if (!this.supplierCertificationForm.valid) {
      this.supplierCertificationForm.markAllAsTouched();
      return;
    }

    const supplierCertification: SupplierCertification = this.supplierCertificationForm.getRawValue();

    this.supplierCertificationService.saveIndividualDocument(supplierCertification).subscribe({
      next: (response: SupplierCertification) => {
        this.toastrService.success(this.translationService.getValue('SUPPLIER_CERTIFICATE_SAVE_SUCCESSFULLY'));
        this.supplierCertification = response;
        this.supplierCertificationForm.reset();
        this.resetFileInput(file);
        this.supplierCertificationForm.get('supplierId')?.setValue(response.supplierId);
      },
    });
  }

  resetFileInput(fileInput: HTMLInputElement): void {
    fileInput.value = '';
  }
}
