import { Component, inject, Input, OnChanges, OnInit, SimpleChanges } from '@angular/core';
import { SupplierCertification } from '../../model/supplier-certification';
import { MatTableModule } from '@angular/material/table';
import { BaseComponent } from '../../../base.component';
import { SupplierCertificationService } from '../supplier-certification.service';

import { ToastrService } from '@core/services/toastr-service';
import { MatDialog } from '@angular/material/dialog';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { DocumentView } from '@core/domain-classes/document-view';
import { BasePreviewComponent } from '@shared/base-preview/base-preview.component';
import { OverlayPanel } from '@shared/overlay-panel/overlay-panel.service';
import { QmsModuleEnum } from '@core/domain-classes/qms-module-enum';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { TranslateModule } from '@ngx-translate/core';
import { ActivatedRoute } from '@angular/router';
import { SupplierCertificationStatusPipe } from '../../pipes/supplier-certification-status.pipe';
import { NgClass } from '@angular/common';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';
import { MatButtonModule } from '@angular/material/button';

@Component({
  selector: 'app-supplier-certification-list',
  imports: [
    MatTableModule,
    HasClaimDirective,
    UTCToLocalTime,
    TranslateModule,
    SupplierCertificationStatusPipe,
    NgClass,
    MatIconModule,
    MatCardModule,
    MatButtonModule
  ],
  templateUrl: './supplier-certification-list.component.html',
  styleUrl: './supplier-certification-list.component.scss'
})
export class SupplierCertificationListComponent extends BaseComponent implements OnInit, OnChanges {
  @Input() supplierId: string | null = null;
  @Input() supplierCertifications: SupplierCertification = {} as SupplierCertification;

  displayedColumns: string[] = ['action', 'fileName', 'certificationName', 'certificationNumber', 'status', 'issudedBy', 'issuedDate', 'expiryDate',];
  supplierCertificationList: SupplierCertification[] = [];
  supplierCertificationService = inject(SupplierCertificationService);
  dialog = inject(MatDialog);
  toastrService = inject(ToastrService);
  commonDialogService = inject(CommonDialogService);
  overlay = inject(OverlayPanel);
  route = inject(ActivatedRoute);

  ngOnInit(): void {
    this.route.parent?.params.subscribe(params => {
      this.supplierId = params['id'];
      this.getSupplierCertification(this.supplierId ?? '');
    });
  }

  ngOnChanges(changes: SimpleChanges): void {
    if (changes['supplierCertifications'] && this.supplierCertifications) {
      this.supplierCertificationList = [...this.supplierCertificationList, this.supplierCertifications]
    }
  }

  getSupplierCertification(supplierId: string) {
    this.sub$.sink = this.supplierCertificationService.getSupplierCertification(supplierId).subscribe({
      next: (response) => {
        const supplierCertifications = response as SupplierCertification[];
        if (supplierCertifications && supplierCertifications.length > 0) {
          this.supplierCertificationList = supplierCertifications;
        } else {
          this.supplierCertificationList = [];
        }
      },
    });
  }

  deleteSupplierCertification(certificateId: string): void {
    this.commonDialogService.deleteConfirmtionDialog(this.translationService.getValue("ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_SUPPLIER_CERTIFICATION")).subscribe({
      next: (result: boolean) => {
        if (result) {
          this.sub$.sink = this.supplierCertificationService.deleteSupplierCertification(certificateId).subscribe({
            next: () => {
              this.toastrService.success(this.translationService.getValue('SUPPLIER_CERTIFICATION_DELETED_SUCCESSFULLY'));
              this.supplierCertificationList = this.supplierCertificationList.filter(certification => certification.id !== certificateId);
            },
          });
        }
      }
    });
  }

  onMeidaView(document: SupplierCertification): void {
    const documentView: DocumentView = {
      documentId: document.id,
      name: document.fileName,
      extension: document.fileName?.split('.').pop() || '',
      isVersion: false,
      isFromPublicPreview: false,
      isPreviewDownloadEnabled: true,
      isFileRequestDocument: false,
      isChunk: false,
      moduleNo: QmsModuleEnum.supplier,
    };
    this.overlay.open(BasePreviewComponent, {
      position: 'center',
      origin: 'global',
      panelClass: ['file-preview-overlay-container', 'white-background'],
      data: documentView,
    });
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.supplierCertificationList.indexOf(row);
  }
}
