import { Component, Inject, inject, OnInit, Optional } from '@angular/core';
import {
  FormBuilder,
  FormGroup,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { SupplierContactListComponent } from '../supplier-contact-list/supplier-contact-list.component';
import { BaseComponent } from '../../../base.component';
import { ActivatedRoute, RouterLink } from '@angular/router';
import { MatCardModule } from '@angular/material/card';
import { MatButtonModule } from '@angular/material/button';
import { MAT_DIALOG_DATA, MatDialogRef } from '@angular/material/dialog';
import { SupplierContact } from '../../model/supplier-contact';
import { NgClass } from '@angular/common';
import { SupplierContactService } from '../supplier-contact.service';

import { ToastrService } from '@core/services/toastr-service';
import { TranslateModule } from '@ngx-translate/core';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { MatIconModule } from '@angular/material/icon';

@Component({
  selector: 'app-manage-supplier-contact',
  imports: [
    ReactiveFormsModule,
    SupplierContactListComponent,
    RouterLink,
    MatCardModule,
    MatButtonModule,
    NgClass,
    TranslateModule,
    HasClaimDirective,
    MatIconModule
  ],
  templateUrl: './manage-supplier-contact.component.html',
  styleUrl: './manage-supplier-contact.component.scss',
})
export class ManageSupplierContactComponent
  extends BaseComponent
  implements OnInit {
  supplierContact: SupplierContact = {} as SupplierContact;
  supplierContactForm: FormGroup;
  fb = inject(FormBuilder);
  supplierContactService = inject(SupplierContactService);
  toastrService = inject(ToastrService);
  route = inject(ActivatedRoute);

  constructor(
    @Optional() private dialogRef: MatDialogRef<ManageSupplierContactComponent>,
    @Optional()
    @Inject(MAT_DIALOG_DATA)
    public data: { supplierContact: SupplierContact; isViewOnly: boolean }
  ) {
    super();
  }

  ngOnInit(): void {
    this.createSupplierContactForm();

    this.route.parent?.params.subscribe(params => {
      const supplierId = params['id'];
      if (supplierId) {
        this.supplierContactForm.get('supplierId')?.setValue(supplierId);
      }
    });

    if (this.data !== null) {
      this.supplierContactForm.patchValue(this.data.supplierContact);
    } else {
      this.data = {
        supplierContact: {} as SupplierContact,
        isViewOnly: false,
      };
    }
  }

  createSupplierContactForm() {
    this.supplierContactForm = this.fb.group({
      id: [''],
      supplierId: [''],
      name: ['', [Validators.required]],
      email: ['', [Validators.required, Validators.email]],
      phone: ['', [Validators.required]],
      position: ['', [Validators.required]],
    });
  }

  onSubmit() {
    if (!this.supplierContactForm.valid) {
      this.supplierContactForm.markAllAsTouched();
      return;
    }
    const supplierContact: SupplierContact =
      this.supplierContactForm.getRawValue();

    this.sub$.sink = this.supplierContactService
      .addSupplierContact(supplierContact)
      .subscribe({
        next: (response: SupplierContact) => {
          this.toastrService.success(
            this.translationService.getValue(
              'SUPPLIER_CONTACT_CREATED_SUCCESSFULLY'
            )
          );
          this.supplierContact = response;
          this.supplierContactForm.reset();
          this.supplierContactForm.get('supplierId')?.setValue(response.supplierId);
        },
      });
  }

  onCancel(): void {
    if (this.dialogRef) {
      this.dialogRef.close();
    }
  }
}
