import { Component, inject, Input, OnChanges, OnInit, SimpleChanges } from '@angular/core';
import { SupplierContact } from '../../model/supplier-contact';
import { BaseComponent } from '../../../base.component';
import { SupplierContactService } from '../supplier-contact.service';
import { MatTableModule } from '@angular/material/table';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';

import { ToastrService } from '@core/services/toastr-service';
import { ManageSupplierContactComponent } from '../manage-supplier-contact/manage-supplier-contact.component';
import { MatDialog } from '@angular/material/dialog';
import { TranslateModule } from '@ngx-translate/core';
import { ActivatedRoute } from '@angular/router';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';
import { NgClass } from '@angular/common';
import { MatButtonModule } from '@angular/material/button';

@Component({
  selector: 'app-supplier-contact-list',
  imports: [
    MatTableModule,
    TranslateModule,
    HasClaimDirective,
    MatIconModule,
    MatCardModule,
    NgClass,
    MatButtonModule
  ],
  templateUrl: './supplier-contact-list.component.html',
  styleUrl: './supplier-contact-list.component.scss'
})
export class SupplierContactListComponent extends BaseComponent implements OnInit, OnChanges {
  @Input() supplierContactResponse: SupplierContact = {} as SupplierContact;

  displayedColumns: string[] = ['action', 'name', 'email', 'phone', 'position'];
  supplierContactService = inject(SupplierContactService);
  commandDialogService = inject(CommonDialogService);
  toastrService = inject(ToastrService);
  dialog = inject(MatDialog);
  route = inject(ActivatedRoute);
  supplierContactList: SupplierContact[] = [];

  ngOnInit(): void {
    this.route.parent?.params.subscribe(params => {
      const supplierId = params['id'];
      if (supplierId) {
        this.getAllSupplierContacts(supplierId);
      }
    });
  }

  ngOnChanges(changes: SimpleChanges): void {
    if (changes['supplierContactResponse'] && this.supplierContactResponse) {
      this.supplierContactList = [...this.supplierContactList, this.supplierContactResponse];
    }
  }

  getAllSupplierContacts(supplierId: string) {
    this.sub$.sink = this.supplierContactService.getSupplierContact(supplierId).subscribe({
      next: (contacts: SupplierContact[]) => {
        const supplierContacts = contacts;
        if (supplierContacts) {
          this.supplierContactList = supplierContacts;
        } else {
          this.supplierContactList = [];
        }
      },
    });
  }

  viewSupplierContact(supplierContact: SupplierContact) {
    this.dialog.open(ManageSupplierContactComponent, {
      width: '500px',
      data: { supplierContact, isViewOnly: true },
    });
  }

  deleteSupplierContact(supplierContactId: string) {
    this.commandDialogService.deleteConfirmtionDialog(this.translationService.getValue("ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_SUPPLIER_CONTACT")).subscribe({
      next: (result: boolean) => {
        if (result) {
          this.sub$.sink = this.supplierContactService.deleteSupplierContact(supplierContactId).subscribe({
            next: () => {
              this.toastrService.success(this.translationService.getValue("SUPPLIER_CONTACT_DELETED_SUCCESSFULLY"));
              this.supplierContactList = this.supplierContactList.filter(supplierContact => supplierContact.id !== supplierContactId);
            },
          });
        };
      }
    });
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.supplierContactList.indexOf(row);
  }
}
