import { NgClass } from '@angular/common';
import {
  AfterViewInit,
  Component,
  inject,
  OnInit,
  ViewChild,
} from '@angular/core';
import { FormsModule } from '@angular/forms';
import { MatButtonModule } from '@angular/material/button';
import { MatIconModule } from '@angular/material/icon';
import { MatMenuModule } from '@angular/material/menu';
import { MatPaginator } from '@angular/material/paginator';
import { MatSelectModule } from '@angular/material/select';
import { MatSort, MatSortModule } from '@angular/material/sort';
import { MatTableModule } from '@angular/material/table';
import { RouterModule } from '@angular/router';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { PageSizeOption } from '@core/utils/global-config';
import { debounceTime, distinctUntilChanged, merge, Subject, tap } from 'rxjs';
import { DepartmentStore } from '../../audit-template/department/store/department.store';
import { BaseComponent } from '../../base.component';
import { UserStore } from '../../user/store/user.store';
import { SupplierStore } from '../supplier-store';
import { Supplier } from '../../supplier/model/supplier';
import { SupplierStatus } from '../../supplier/model/supplier-status';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { SupplierStatusPipe } from '../pipes/supplier-status.pipe';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-supplier-list',
  imports: [
    RouterModule,
    MatTableModule,
    MatButtonModule,
    MatMenuModule,
    MatIconModule,
    MatPaginator,
    MatSelectModule,
    MatSortModule,
    PageHelpTextComponent,
    HasClaimDirective,
    FormsModule,
    TranslateModule,
    NgClass,
    SupplierStatusPipe,
    MatCardModule
  ],
  templateUrl: './supplier-list.component.html',
  styleUrl: './supplier-list.component.scss',
})
export class SupplierListComponent
  extends BaseComponent
  implements OnInit, AfterViewInit {
  supplierList: Supplier[] = [];
  displayedColumns: string[] = [
    'action',
    'supplierCode',
    'name',
    'contactPerson',
    'email',
    'phone',
    'country',
    'status',
    'address',
  ];
  displayedColumnSecondary: string[] = [
    'search-action',
    'search-supplierCode',
    'search-name',
    'search-contactPerson',
    'search-email',
    'search-phone',
    'search-country',
    'search-status',
    'search-address',
  ];
  footerToDisplayed: string[] = ['footer'];
  supplierStore = inject(SupplierStore);
  pageOption = PageSizeOption;
  userStore = inject(UserStore);
  commandDialogService = inject(CommonDialogService);
  departmentStore = inject(DepartmentStore);
  supplierFilterParameter = { ...this.supplierStore.filterParameters() };
  filterParameter$: Subject<string> = new Subject<string>();
  _supplierCodeFilter = this.supplierFilterParameter.supplierCode;
  _nameFilter = this.supplierFilterParameter.name;
  _contactPersonFilter = this.supplierFilterParameter.contactPerson;
  _emailFilter = this.supplierFilterParameter.email;
  _phoneFilter = this.supplierFilterParameter.phone;
  _countryFilter = this.supplierFilterParameter.country;
  _statusFilter = this.supplierFilterParameter.status;

  @ViewChild(MatPaginator) paginator: MatPaginator;
  @ViewChild(MatSort) sort: MatSort;

  statusList = Object.keys(SupplierStatus)
    .filter((key) => !isNaN(Number(SupplierStatus[key as any])))
    .map((key) => ({
      label: key,
      value: SupplierStatus[key as keyof typeof SupplierStatus],
    }));

  public get supplierCodeFilter(): string {
    return this._supplierCodeFilter;
  }

  public set supplierCodeFilter(v: string) {
    if (this._supplierCodeFilter !== v) {
      this._supplierCodeFilter = v;
      const supplierCodeFilter = `supplierCode#${v}`;
      this.filterParameter$.next(supplierCodeFilter);
    }
  }

  public get nameFilter(): string {
    return this._nameFilter;
  }

  public set nameFilter(v: string) {
    if (this._nameFilter !== v) {
      this._nameFilter = v;
      const nameFilter = `name#${v}`;
      this.filterParameter$.next(nameFilter);
    }
  }

  public get contactPersonFilter(): string {
    return this._contactPersonFilter;
  }

  public set contactPersonFilter(v: string) {
    if (this._contactPersonFilter !== v) {
      this._contactPersonFilter = v;
      const contactPersonFilter = `contactPerson#${v}`;
      this.filterParameter$.next(contactPersonFilter);
    }
  }

  public get emailFilter(): string {
    return this._emailFilter;
  }

  public set emailFilter(v: string) {
    if (this._emailFilter !== v) {
      this._emailFilter = v;
      const emailFilter = `email#${v}`;
      this.filterParameter$.next(emailFilter);
    }
  }

  public get phoneFilter(): string {
    return this._phoneFilter;
  }

  public set phoneFilter(v: string) {
    if (this._phoneFilter !== v) {
      this._phoneFilter = v;
      const phoneFilter = `phone#${v}`;
      this.filterParameter$.next(phoneFilter);
    }
  }

  public get countryFilter(): string {
    return this._countryFilter ?? '';
  }

  public set countryFilter(v: string) {
    if (this._countryFilter !== v) {
      this._countryFilter = v;
      const countryFilter = `country#${v}`;
      this.filterParameter$.next(countryFilter);
    }
  }

  public get statusFilter(): string {
    return this._statusFilter ?? '';
  }

  public set statusFilter(v: string) {
    if (this._statusFilter !== v) {
      this._statusFilter = v;
      const statusFilter = `status#${v}`;
      this.filterParameter$.next(statusFilter);
    }
  }

  ngOnInit(): void {
    this.sub$.sink = this.filterParameter$
      .pipe(debounceTime(1000), distinctUntilChanged())
      .subscribe((c: string) => {
        this.supplierFilterParameter.skip = 0;
        this.paginator.pageIndex = 0;
        const filterArray: Array<string> = c.split('#');
        if (filterArray[0] === 'supplierCode') {
          this.supplierFilterParameter.supplierCode = filterArray[1];
        } else if (filterArray[0] === 'name') {
          this.supplierFilterParameter.name = filterArray[1];
        } else if (filterArray[0] === 'contactPerson') {
          this.supplierFilterParameter.contactPerson = filterArray[1];
        } else if (filterArray[0] === 'email') {
          this.supplierFilterParameter.email = filterArray[1];
        } else if (filterArray[0] === 'phone') {
          this.supplierFilterParameter.phone = filterArray[1];
        } else if (filterArray[0] === 'country') {
          this.supplierFilterParameter.country = filterArray[1];
        } else if (filterArray[0] === 'status') {
          this.supplierFilterParameter.status = filterArray[1];
        }
        this.supplierStore.loadByQuery(this.supplierFilterParameter);
      });
  }

  getDepartments(): void {
    this.departmentStore.loadDepartments();
  }

  refresh() {
    this.supplierStore.loadByQuery(this.supplierFilterParameter);
  }

  deleteSupplier(id: string) {
    this.commandDialogService
      .deleteConfirmtionDialog(
        this.translationService.getValue(
          'ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_SUPPLIER'
        )
      )
      .subscribe({
        next: (result: boolean) => {
          if (result) {
            this.supplierStore.deleteSupplierById(id);
          }
        },
      });
  }

  ngAfterViewInit(): void {
    this.sub$.sink = this.sort.sortChange.subscribe(() => {
      this.paginator.pageIndex = 0;
    });
    this.sub$.sink = merge(this.sort.sortChange, this.paginator.page)
      .pipe(
        tap(() => {
          this.supplierFilterParameter.skip =
            this.paginator.pageIndex * this.paginator.pageSize;
          this.supplierFilterParameter.pageSize = this.paginator.pageSize;
          this.supplierFilterParameter.orderBy =
            this.sort.active + ' ' + this.sort.direction;
          this.supplierStore.loadByQuery(this.supplierFilterParameter);
        })
      )
      .subscribe();
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.supplierStore.supplierList().indexOf(row);
  }
}
