import { NgClass } from '@angular/common';
import { Component, Inject, inject, OnInit, Optional } from '@angular/core';
import { FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { MatCardModule } from '@angular/material/card';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { MatSelectModule } from '@angular/material/select';
import { BaseComponent } from '../../../base.component';
import { SupplierPerformance } from '../../model/supplier-performance';
import { MAT_DIALOG_DATA, MatDialogRef } from '@angular/material/dialog';
import { UserStore } from '../../../user/store/user.store';
import { MatButtonModule } from '@angular/material/button';
import { TextEditorComponent } from '@shared/text-editor/text-editor.component';
import { SupplierPerformanceListComponent } from '../supplier-performance-list/supplier-performance-list.component';
import { SupplierPerformanceService } from '../supplier-performance.service';

import { ToastrService } from '@core/services/toastr-service';
import { ActivatedRoute, RouterLink } from '@angular/router';
import { SecurityService } from '@core/security/security.service';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { MatIconModule } from '@angular/material/icon';

@Component({
  selector: 'app-manage-supplier-performance',
  imports: [
    ReactiveFormsModule,
    MatSelectModule,
    MatDatepickerModule,
    NgClass,
    MatCardModule,
    MatButtonModule,
    TextEditorComponent,
    SupplierPerformanceListComponent,
    RouterLink,
    HasClaimDirective,
    TranslateModule,
    MatIconModule
  ],
  templateUrl: './manage-supplier-performance.component.html',
  styleUrl: './manage-supplier-performance.component.scss'
})
export class ManageSupplierPerformanceComponent extends BaseComponent implements OnInit {
  supplierPerformanceForm: FormGroup;
  fb = inject(FormBuilder);
  supplierPerformance: SupplierPerformance = {} as SupplierPerformance;
  userStore = inject(UserStore);
  supplierPerformanceService = inject(SupplierPerformanceService);
  securityService = inject(SecurityService);
  toastrService = inject(ToastrService);
  route = inject(ActivatedRoute);

  constructor(
    @Optional() private dialogRef: MatDialogRef<ManageSupplierPerformanceComponent>,
    @Optional() @Inject(MAT_DIALOG_DATA) public data: { supplierPerformance: SupplierPerformance, isViewOnly: boolean },
  ) {
    super();
  }

  ngOnInit(): void {
    this.createSupplierPerformanceForm();

    this.route.parent?.paramMap.subscribe((params) => {
      const supplierId = params.get('id');
      if (supplierId) {
        this.supplierPerformanceForm.get('supplierId')?.setValue(supplierId);
      }
    });

    if (this.data !== null) {
      this.supplierPerformanceForm.patchValue(this.data.supplierPerformance);
    } else {
      this.data = {
        supplierPerformance: {} as SupplierPerformance,
        isViewOnly: false
      }
    }
  }

  createSupplierPerformanceForm() {
    this.supplierPerformanceForm = this.fb.group({
      id: [''],
      supplierId: [''],
      evaluationDate: [new Date(), [Validators.required]],
      criteria: ['', [Validators.required]],
      score: ['', [Validators.required, Validators.min(0), Validators.max(100)]],
      evaluatedById: [this.securityService.securityObject.id, [Validators.required]],
      remarks: ['', [Validators.required]],
    });
  }

  onSubmit() {
    if (this.supplierPerformanceForm.invalid) {
      this.supplierPerformanceForm.markAllAsTouched();
      return;
    }

    const supplierPerformance: SupplierPerformance = this.supplierPerformanceForm.getRawValue();

    if (!supplierPerformance.supplierId) {
      this.toastrService.error(this.translationService.getValue('SUPPLIER_ID_IS_REQUIRED'));
      return;
    }

    this.sub$.sink = this.supplierPerformanceService.addSupplierPerformance(supplierPerformance).subscribe({
      next: (response: SupplierPerformance) => {
        this.toastrService.success(this.translationService.getValue('SUPPLIER_PERFORMANCE_CREATED_SUCCESSFULLY'));
        this.supplierPerformance = response;
        this.supplierPerformanceForm.reset({
          supplierId: response.supplierId,
          evaluationDate: new Date(),
          evaluatedById: this.securityService.securityObject.id
        });
      },
    });
  }

  onCancel() {
    if (this.dialogRef) {
      this.dialogRef.close();
    }
  }
}
