import { NgClass } from '@angular/common';
import { Component, inject, Inject, OnInit, Optional } from '@angular/core';
import {
  FormBuilder,
  FormGroup,
  ReactiveFormsModule,
  Validators,
} from '@angular/forms';
import { MatCardModule } from '@angular/material/card';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { MatSelectModule } from '@angular/material/select';
import { SupplierQualification } from '../../model/supplier-qualification';
import { MatDialogRef, MAT_DIALOG_DATA } from '@angular/material/dialog';
import { BaseComponent } from '../../../base.component';
import { UserStore } from '../../../user/store/user.store';
import { TextEditorComponent } from '@shared/text-editor/text-editor.component';
import { SupplierQualificationService } from '../supplier-qualification.service';
import { SupplierQualificationListComponent } from '../supplier-qualification-list/supplier-qualification-list.component';
import { ActivatedRoute, RouterLink } from '@angular/router';
import { MatButtonModule } from '@angular/material/button';
import { SUPPLIER_QUALIFICATION_RESULT } from '../../model/supplier-qualification.enum';
import { SecurityService } from '@core/security/security.service';

import { ToastrService } from '@core/services/toastr-service';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { MatIconModule } from '@angular/material/icon';

@Component({
  selector: 'app-manage-supplier-qualification',
  imports: [
    MatSelectModule,
    MatDatepickerModule,
    NgClass,
    MatCardModule,
    ReactiveFormsModule,
    TextEditorComponent,
    SupplierQualificationListComponent,
    RouterLink,
    MatButtonModule,
    HasClaimDirective,
    TranslateModule,
    MatIconModule
  ],
  templateUrl: './manage-supplier-qualification.component.html',
  styleUrl: './manage-supplier-qualification.component.scss',
})
export class ManageSupplierQualificationComponent
  extends BaseComponent
  implements OnInit {
  supplierQualificationForm: FormGroup;
  fb = inject(FormBuilder);
  userStore = inject(UserStore);
  supplierQualificationService = inject(SupplierQualificationService);
  toastrService = inject(ToastrService);
  securityService = inject(SecurityService);
  supplierQualification: SupplierQualification = {} as SupplierQualification;
  minQualificationDate: Date = new Date();
  route = inject(ActivatedRoute);

  supplierResultList = Object.keys(SUPPLIER_QUALIFICATION_RESULT)
    .filter((key) => isNaN(Number(key)))
    .map((key) => ({
      label: key,
      value:
        SUPPLIER_QUALIFICATION_RESULT[
        key as keyof typeof SUPPLIER_QUALIFICATION_RESULT
        ],
    }));

  constructor(
    @Optional()
    private dialogRef: MatDialogRef<ManageSupplierQualificationComponent>,
    @Optional()
    @Inject(MAT_DIALOG_DATA)
    public data: {
      supplierQualification: SupplierQualification;
      isViewOnly: boolean;
    }
  ) {
    super();
  }

  ngOnInit(): void {
    this.createSupplierQualificationForm();

    this.route.parent?.paramMap.subscribe((params) => {
      const supplierId = params.get('id');
      if (supplierId) {
        this.supplierQualificationForm.get('supplierId')?.setValue(supplierId);
      }
    });

    if (this.data !== null) {
      this.supplierQualificationForm.patchValue(
        this.data.supplierQualification
      );
    } else {
      this.data = {
        supplierQualification: {} as SupplierQualification,
        isViewOnly: false
      }
    }
  }

  createSupplierQualificationForm() {
    this.supplierQualificationForm = this.fb.group({
      id: [''],
      supplierId: [''],
      qualificationDate: [null, [Validators.required]],
      qualifiedById: [
        this.securityService.securityObject.id,
        [Validators.required],
      ],
      criteria: ['', [Validators.required]],
      result: ['', [Validators.required]],
      remarks: ['', [Validators.required]],
    });
  }

  onSubmit() {
    if (!this.supplierQualificationForm.valid) {
      this.supplierQualificationForm.markAllAsTouched();
      return;
    }

    const supplierQualification: SupplierQualification =
      this.supplierQualificationForm.getRawValue();

    this.sub$.sink = this.supplierQualificationService
      .addSupplierQualification(supplierQualification)
      .subscribe({
        next: (response: SupplierQualification) => {
          this.toastrService.success(
            this.translationService.getValue(
              'SUPPLIER_QUALIFICATION_CRATED_SUCCESSFULLY'
            )
          );
          this.supplierQualification = response;
          this.supplierQualificationForm.reset();
          this.supplierQualificationForm.get('supplierId')?.setValue(
            response.supplierId
          );
        },
      });
  }

  onCancel() {
    this.dialogRef.close();
  }
}
