import {
  patchState,
  signalStore,
  withHooks,
  withMethods,
  withState,
} from '@ngrx/signals';
import { rxMethod } from '@ngrx/signals/rxjs-interop';
import { inject } from '@angular/core';

import { ToastrService } from '@core/services/toastr-service';
import { TranslationService } from '@core/services/translation.service';
import { distinctUntilChanged, pipe, switchMap, tap } from 'rxjs';
import { tapResponse } from '@ngrx/operators';
import { toObservable } from '@angular/core/rxjs-interop';
import { HttpResponse } from '@angular/common/http';
import { Router } from '@angular/router';
import { Supplier } from '../supplier/model/supplier';
import { SupplierResource } from './model/supplier-resource';
import { SupplierService } from './supplier.service';

type SupplierState = {
  supplierList: Supplier[];
  supplier: Supplier;
  loadList: boolean;
  isAddUpdate: boolean;
  filterParameters: SupplierResource;
};

export const initialSupplierState: SupplierState = {
  supplierList: [],
  supplier: {} as Supplier,
  loadList: true,
  isAddUpdate: false,
  filterParameters: {
    orderBy: 'supplierCode asc',
    pageSize: 10,
    skip: 0,
    totalCount: 0,
    name: '',
    supplierCode: '',
    contactPerson: '',
    email: '',
    phone: '',
    country: '',
    status: '',
  },
};

export const SupplierStore = signalStore(
  { providedIn: 'root' },
  withState(initialSupplierState),
  withMethods(
    (
      store,
      supplierService = inject(SupplierService),
      router = inject(Router),
      toastrService = inject(ToastrService),
      translationService = inject(TranslationService)
    ) => ({
      loadByQuery: rxMethod<SupplierResource>(
        pipe(
          tap(() => patchState(store, { supplierList: [] })),
          switchMap((filter) => {
            return supplierService.getSuppliers(filter).pipe(
              tapResponse({
                next: (httpResponse: HttpResponse<Supplier[]>) => {
                  if (httpResponse && httpResponse.body) {
                    patchState(store, {
                      supplierList: [...httpResponse.body],
                      loadList: false,
                    });
                  }
                  if (httpResponse.headers.get('x-pagination')) {
                    const pagination: SupplierResource = JSON.parse(
                      httpResponse.headers.get('x-pagination') ?? ''
                    );
                    patchState(store, {
                      filterParameters: {
                        ...filter,
                        totalCount: pagination.totalCount,
                      },
                    });
                  }
                },
                error: (error: any) => {
                  toastrService.error(error.error);
                },
              })
            );
          })
        )
      ),
      addSupplier: rxMethod<Supplier>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { loadList: false })),
          switchMap((supplier: Supplier) =>
            supplierService.addSupplier(supplier).pipe(
              tapResponse({
                next: (addSupplier: Supplier) => {
                  toastrService.success(
                    translationService.getValue('SUPPLIER_CREATED_SUCCESSFULLY')
                  );
                  patchState(store, { isAddUpdate: true, loadList: true });
                },
                error: (err: any) => {
                  patchState(store, { loadList: true });
                },
              })
            )
          )
        )
      ),
      updateSupplier: rxMethod<Supplier>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { loadList: false })),
          switchMap((updateSupplier: Supplier) =>
            supplierService.updateSupplier(updateSupplier).pipe(
              tapResponse({
                next: (response) => {
                  const updatedSupplier: Supplier = response as Supplier;
                  toastrService.success(
                    translationService.getValue('SUPPLIER_UPDATED_SUCCESSFULLY')
                  );
                  patchState(store, {
                    supplierList: store.supplierList().map((supplier) => {
                      return updatedSupplier.id === supplier.id
                        ? updatedSupplier
                        : supplier;
                    }),
                    isAddUpdate: true,
                  });
                },
                error: (err: any) => {
                  patchState(store, { loadList: true });
                },
              })
            )
          )
        )
      ),
      deleteSupplierById: rxMethod<string>(
        pipe(
          distinctUntilChanged(),
          tap(() => patchState(store, { loadList: false })),
          switchMap((supplierId: string) =>
            supplierService.deleteSupplier(supplierId).pipe(
              tapResponse({
                next: () => {
                  toastrService.success(
                    translationService.getValue('SUPPLIER_DELETED_SUCCESSFULLY')
                  );
                  patchState(store, {
                    supplierList: store
                      .supplierList()
                      .filter((w) => w.id !== supplierId),
                    filterParameters: {
                      ...store.filterParameters(),
                      totalCount: store.filterParameters().totalCount - 1,
                    },
                  });
                },
                error: (err: any) => {
                  console.error(err);
                },
              })
            )
          )
        )
      ),
      resetflag() {
        patchState(store, { isAddUpdate: false });
      },
    })
  ),
  withHooks({
    onInit(store) {
      toObservable(store.loadList).subscribe((flag) => {
        if (flag) {
          store.loadByQuery(store.filterParameters());
        }
      });
    },
  })
);
