import { HttpClient, HttpParams, HttpResponse } from '@angular/common/http';
import { Injectable } from '@angular/core';
import { CommonError } from '../core/error-handler/common-error';
import { CommonHttpErrorService } from '../core/error-handler/common-http-error.service';
import { Observable } from 'rxjs';
import { catchError } from 'rxjs/operators';
import { CommonService } from '../core/services/common.service';
import { Supplier } from '../supplier/model/supplier';
import { SupplierResource } from './model/supplier-resource';

@Injectable({
  providedIn: 'root',
})
export class SupplierService {
  constructor(
    private httpClient: HttpClient,
    private commonHttpErrorService: CommonHttpErrorService,
    private commonService: CommonService
  ) { }

  getSuppliers(
    resource: SupplierResource
  ): Observable<HttpResponse<Supplier[]>> {
    const url = 'Supplier';
    const customParams = new HttpParams()
      .set('OrderBy', resource.orderBy)
      .set('PageSize', resource.pageSize)
      .set('Skip', resource.skip)
      .set('Name', resource.name)
      .set('SupplierCode', resource.supplierCode)
      .set('ContactPerson', resource.contactPerson ?? '')
      .set('Email', resource.email ?? '')
      .set('Phone', resource.phone ?? '')
      .set('Country', resource.country ?? '')
      .set('Status', resource.status ?? '');
    return this.httpClient
      .get<Supplier[]>(url, {
        params: customParams,
        observe: 'response',
      });

  }

  getSupplier(id: string): Observable<Supplier | CommonError> {
    const url = `Supplier/${id}`;
    return this.httpClient
      .get<Supplier>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  addSupplier(supplier: Supplier): Observable<Supplier> {
    const url = `Supplier`;
    return this.httpClient
      .post<Supplier>(url, supplier);

  }

  updateSupplier(supplier: Supplier): Observable<Supplier | CommonError> {
    const url = `Supplier/${supplier.id}`;
    return this.httpClient
      .put<Supplier>(url, supplier)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  deleteSupplier(id: string): Observable<Supplier | CommonError> {
    const url = `Supplier/${id}`;
    return this.httpClient
      .delete<Supplier>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getSupplierCode(): Observable<Supplier | CommonError> {
    const url = 'Supplier/GetNewSupplierCode';
    return this.httpClient
      .get<Supplier>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }
}
