import { inject, Injectable } from '@angular/core';
import { Session } from '@core/domain-classes/session';
import { CommonError } from '@core/error-handler/common-error';
import { CommonHttpErrorService } from '@core/error-handler/common-http-error.service';
import { HttpClient } from '@angular/common/http';
import { catchError, Observable } from 'rxjs';
import { EmployeeCourseSessionQuiz } from '@core/domain-classes/employee-course-session-quiz';
import { QuizQuestion } from '@core/domain-classes/session-quiz-question';
import { SessionQuiz } from '@core/domain-classes/session-quiz';


@Injectable({
  providedIn: 'root'
})
export class SessionService {
  private httpClient = inject(HttpClient);
  private commonHttpErrorService = inject(CommonHttpErrorService);

  constructor() { }

  getSession(id: string): Observable<Session | CommonError> {
    const url = `CourseSession/${id}`
    return this.httpClient
      .get<Session>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getCourseSession(id: string): Observable<Session[] | CommonError> {
    const url = `CourseSession/${id}/sessions`
    return this.httpClient
      .get<Session[]>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  addSession(session: Session): Observable<Session | CommonError> {
    const url = 'CourseSession';
    return this.httpClient
      .post<Session>(url, session)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  updateSession(session: Session): Observable<Session | CommonError> {
    const url = `CourseSession/${session.id}`
    return this.httpClient
      .put<Session>(url, session)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  deleteSession(id: string): Observable<void | CommonError> {
    const url = `CourseSession/${id}`
    return this.httpClient
      .delete<void>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  addSessionQuizQuestion(quizQuestion: QuizQuestion): Observable<QuizQuestion | CommonError> {
    const url = 'CourseSessionQuizQuestion'
    return this.httpClient
      .post<QuizQuestion>(url, quizQuestion)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getSessionQuizQuestion(sessionId: string): Observable<SessionQuiz | CommonError> {
    const url = `CourseSessionQuiz/${sessionId}`
    return this.httpClient
      .get<SessionQuiz>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  updateQuizQuestion(question: QuizQuestion): Observable<QuizQuestion | CommonError> {
    const url = `CourseSessionQuizQuestion/${question.id}`
    return this.httpClient
      .put<QuizQuestion>(url, question)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  deleteQuizQuestion(questionId: string): Observable<void | CommonError> {
    const url = `CourseSessionQuizQuestion/${questionId}`
    return this.httpClient
      .delete<void>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  updatePassingScore(courseSessionId: string, passingScore: number): Observable<SessionQuiz | CommonError> {
    const url = `CourseSessionQuiz/${courseSessionId}`
    return this.httpClient
      .put<SessionQuiz>(url, { passingScore, sessionId: courseSessionId })
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }
}
