import { Component, Inject, inject, OnInit } from '@angular/core';
import { MAT_DIALOG_DATA, MatDialogModule, MatDialogRef } from '@angular/material/dialog';
import { TranslateModule } from '@ngx-translate/core';
import { FormArray, FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { environment } from '@environments/environment';
import { CourseSessionMedia } from '@core/domain-classes/course-session-media';

import { ToastrService } from '@core/services/toastr-service';
import { TranslationService } from '@core/services/translation.service';
import { bufferCount, concatMap, from, mergeMap, Observable, tap } from 'rxjs';
import { CommonError } from '@core/error-handler/common-error';
import { CourseSessionMediaChunk } from '@core/domain-classes/course-session-media-chunk';
import { CommonService } from '@core/services/common.service';
import { SessionMediaService } from '../session-meida.service';
import { MediaType, MediaTypeList } from '@core/domain-classes/session-media-type-enum';
import { StorageSetting } from '../../../../storage-setting/storage-setting';
import { StorageSettingService } from '../../../../storage-setting/storage-setting.service';
import { MatButtonModule } from '@angular/material/button';
import { MatIconModule } from '@angular/material/icon';

@Component({
  selector: 'app-add-media',
  imports: [
    TranslateModule,
    ReactiveFormsModule,
    MatDialogModule,
    MatButtonModule,
    MatIconModule
  ],
  templateUrl: './add-media.component.html',
  styleUrl: './add-media.component.scss'
})
export class AddMediaComponent implements OnInit {
  sessionMediaForm!: FormGroup;
  fb = inject(FormBuilder);
  isLink = true
  mediaTypeList = MediaTypeList;
  fileData: any;
  extension = '';
  chunkSize = environment.chunkSize;
  sessionMediaId: string | undefined;
  sessionMedia?: CourseSessionMedia;
  toastrService = inject(ToastrService);
  translationService = inject(TranslationService);
  isLoading = false;
  progress: number = 0;
  commonService = inject(CommonService);
  sessionMediaService = inject(SessionMediaService);
  storageSettingService = inject(StorageSettingService);
  allStorageSetting: StorageSetting<any>[] = [];

  constructor(
    public dialogRef: MatDialogRef<AddMediaComponent>,
    @Inject(MAT_DIALOG_DATA) public data: { courseSessionId: string },
  ) { }

  ngOnInit(): void {
    this.creatMediaForm();
    this.sessionMediaForm.get('courseSessionId')?.setValue(this.data.courseSessionId);
    this.getStorageSetting();
  }


  getStorageSetting() {
    this.storageSettingService.getStorageSettings().subscribe((c: StorageSetting<any>[]) => {
      if (Array.isArray(c)) {
        this.allStorageSetting = c;
        const isDefaultItem = this.allStorageSetting.find((c) => c.isDefault);
        if (isDefaultItem) {
          this.sessionMediaForm.patchValue({ storageSettingId: isDefaultItem.id });
        }
      }
    });
  }

  creatMediaForm() {
    this.sessionMediaForm = this.fb.group({
      id: [''],
      name: ['', [Validators.required]],
      courseSessionId: [''],
      url: ['', [Validators.required]],
      mediaType: [0],
      description: [''],
      storageSettingId: [''],
    })
  }

  async upload(files: FileList | null) {
    if (files == null) return;
    if (files?.length === 0) return;

    this.extension = files[0].name.split('.').pop() ?? '';

    this.fileData = files[0];
    this.sessionMediaForm?.get('url')!.setValue(files[0].name);
    this.sessionMediaForm?.get('name')!.setValue(files[0].name);
  }

  saveDocument() {
    if (this.sessionMediaForm.valid) {
      const sessionMedia: CourseSessionMedia = this.buildDocumentObject();
      if (sessionMedia.mediaType == MediaType.FILE && !this.fileData) {
        this.toastrService.error(this.translationService.getValue('PLEASE_SELECT_FILE'));
        return;
      }
      if (this.fileData && this.fileData.size > this.chunkSize) {
        this.saveDocumentChunk();
      } else {
        this.saveIndividualDocument();
      }
    } else {
      this.markFormGroupTouched(this.sessionMediaForm);
    }
  }

  saveDocumentChunk() {
    this.sessionMediaService
      .addChunkDocument(this.buildDocumentObject())
      .subscribe((c) => {
        const sessionMedia = c as CourseSessionMedia;
        if (sessionMedia && sessionMedia.id) {
          this.sessionMediaId = sessionMedia.id;
          this.sessionMedia = sessionMedia;
          this.uploadFileInChunks(sessionMedia.id);
        }
      });
  }

  uploadFileInChunks(courseSessionMediaId: string) {
    if (!this.fileData) return;
    const { chunkSize1, parallelCalls } = this.commonService.getNetworkSpeed();
    const totalChunks = Math.ceil(this.fileData.size / this.chunkSize);
    const chunkUploads: FormData[] = [];
    for (let i = 0; i < totalChunks; i++) {
      const start = i * this.chunkSize;
      const end = Math.min(start + this.chunkSize, this.fileData.size);
      const chunk = this.fileData.slice(start, end);
      const formData = new FormData();
      formData.append('file', chunk);
      formData.append('chunkIndex', i.toString());
      formData.append('size', this.chunkSize.toString());
      formData.append('totalChunks', totalChunks.toString());
      formData.append('extension', this.extension);
      formData.append('courseSessionMediaId', courseSessionMediaId);
      chunkUploads.push(formData);
    }
    this.isLoading = true;
    from(chunkUploads)
      .pipe(
        bufferCount(parallelCalls), // Group chunks in batches based on parallelCalls
        concatMap(
          (
            batch // Change concatMap to mergeMap
          ) =>
            from(batch).pipe(
              mergeMap((formData) => this.uploadChunk(formData), parallelCalls) // Execute uploads in parallel
            )
        )
      )
      .subscribe({
        next: (data: any) => {
          this.progress = Math.min(
            this.progress + 100 / chunkUploads.length,
            100
          );
        },
        complete: () => {
          this.progress = 100;
          this.isLoading = false;
          this.markChunkAsUploaded();
        },
        error: (err) => {
          this.isLoading = false;
          this.markChunkAsUploaded(false);
        },
      });
  }

  uploadChunk(formData: FormData): Observable<CourseSessionMediaChunk | CommonError> {
    return this.sessionMediaService.uploadChunkDocument(formData);
  }

  markChunkAsUploaded(flag: boolean = true) {
    this.isLoading = true;
    this.sessionMediaService
      .markChunkAsUploaded(this.sessionMediaId, flag)
      .subscribe({
        next: () => {
          this.isLoading = false;
          this.toastrService.success(
            this.translationService.getValue('MEDIA_SAVE_SUCCESSFULLY')
          );
          this.dialogRef.close(true);
        },
        error: () => {
          this.isLoading = false;
        },
      });
  }

  saveIndividualDocument() {
    if (this.sessionMediaForm.valid) {
      const document = this.buildDocumentObject();
      this.sessionMediaService
        .saveIndividualDocument(document)
        .subscribe(() => {
          this.toastrService.success(
            this.translationService.getValue('MEDIA_SAVE_SUCCESSFULLY')
          );
          this.dialogRef.close(true);
        });
    } else {
      this.markFormGroupTouched(this.sessionMediaForm);
    }
  }

  buildDocumentObject(): CourseSessionMedia {
    const document: CourseSessionMedia = {
      storageSettingId: this.sessionMediaForm?.get('storageSettingId')?.value ?? '',
      description: this.sessionMediaForm?.get('description')?.value,
      name: this.sessionMediaForm?.get('name')?.value,
      url: this.sessionMediaForm?.get('url')?.value,
      file: this.fileData,
      extension: this.extension,
      courseSessionId: this.sessionMediaForm?.get('courseSessionId')?.value,
      mediaType: this.sessionMediaForm?.get('mediaType')?.value,
      id: '',
    };
    return document;
  }

  onNoClick(): void {
    this.dialogRef.close();
  }

  private markFormGroupTouched(formGroup: FormGroup) {
    (Object).values(formGroup.controls).forEach((control) => {
      control.markAsTouched();

      if ((control as FormArray).controls) {
        this.markFormGroupTouched(control as FormGroup);
      }
    });
  }
}
