import { Component, inject } from '@angular/core';
import { MatMenuModule } from '@angular/material/menu';
import { MatTableModule } from '@angular/material/table';
import { TranslateModule } from '@ngx-translate/core';
import { MatDialog } from '@angular/material/dialog';
import { AddMediaComponent } from './add-media/add-media.component';
import { ActivatedRoute, RouterLink } from '@angular/router';
import { SessionMediaService } from './session-meida.service';
import { CourseSessionMedia } from '@core/domain-classes/course-session-media';
import { MediaTypePipe } from './session-media-type.pipe';

import { ToastrService } from '@core/services/toastr-service';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { OverlayPanel } from '@shared/overlay-panel/overlay-panel.service';
import { BasePreviewComponent } from '@shared/base-preview/base-preview.component';
import { BaseComponent } from '../../../base.component';
import { DocumentView } from '@core/domain-classes/document-view';
import { DocumentInfo } from '@core/domain-classes/document-info';
import { SecurityService } from '@core/security/security.service';
import { DocumentStore } from '../../../document/document-list/document-store';
import { DocumentResource } from '@core/domain-classes/document-resource';
import { QmsModuleEnum } from '@core/domain-classes/qms-module-enum';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { MatSort } from '@angular/material/sort';
import { MatButtonModule } from '@angular/material/button';
import { MatIconModule } from '@angular/material/icon';
import { NgClass } from '@angular/common';

@Component({
  selector: 'app-session-media',
  imports: [
    TranslateModule,
    MatTableModule,
    MatMenuModule,
    MediaTypePipe,
    PageHelpTextComponent,
    HasClaimDirective,
    MatSort,
    MatButtonModule,
    RouterLink,
    MatIconModule,
    NgClass
  ],
  templateUrl: './session-media.component.html',
  styleUrl: './session-media.component.scss'
})
export class SessionMediaComponent extends BaseComponent {
  courseSessionMedia: CourseSessionMedia[] = [];
  displayedColumns = ['action', 'name', 'mediaType'];
  dialog = inject(MatDialog);
  sessionId: string = '';
  courseId: string = '';
  route = inject(ActivatedRoute);
  sessionMediaService = inject(SessionMediaService);
  toastrService = inject(ToastrService);
  commonDialogService = inject(CommonDialogService);
  securityService = inject(SecurityService);
  documentStore = inject(DocumentStore);
  documentResource: DocumentResource = this.documentStore.documentResourceParameter();

  constructor(public overlay: OverlayPanel) {
    super();
  }

  ngOnInit() {
    this.route.parent?.params.subscribe(params => {
      this.sessionId = params['sessionId'];
      this.courseId = params['courseId'];
      if (this.sessionId) {
        this.getSessionMedia();
      }
    });
  }

  onMeidaView(document: DocumentInfo) {
    const documentView: DocumentView = {
      id: document.id,
      name: document.name,
      url: document.url,
      mediaType: document.mediaType,
      extension: document.extension,
      isVersion: false,
      isFromPublicPreview: false,
      isPreviewDownloadEnabled: true,
      isFileRequestDocument: false,
      isChunk: document.isChunk,
      moduleNo: QmsModuleEnum.Training,
    };
    this.overlay.open(BasePreviewComponent, {
      position: 'center',
      origin: 'global',
      panelClass: ['file-preview-overlay-container', 'white-background'],
      data: documentView,
    });
  }

  getSessionMedia() {
    this.sessionMediaService.getSessionMedia(this.sessionId).subscribe({
      next: (res) => {
        this.courseSessionMedia = res as CourseSessionMedia[];
      }
    })
  }

  manageMedia(): void {
    const dialogRef = this.dialog.open(AddMediaComponent, {
      maxWidth: '40vw',
      width: '100%',
      maxHeight: '90vh',
      data: {
        courseSessionId: this.sessionId
      },
    });

    dialogRef.afterClosed().subscribe((result: boolean) => {
      if (result) {
        this.getSessionMedia();
      }
    });
  }

  deleteMedia(media: CourseSessionMedia) {
    this.commonDialogService.deleteConfirmtionDialog(
      this.translationService.getValue('ARE_YOU_SURE_YOU_WANT_TO_DELETE'))
      .subscribe((res: boolean) => {
        if (res) {
          this.sessionMediaService.deleteSessionMedia(media.id!).subscribe({
            next: (res) => {
              if (res) {
                this.toastrService.success(this.translationService.getValue('MEDIA_DELETED_SUCCESSFULLY'));
                this.courseSessionMedia = this.courseSessionMedia.filter(item => item.id !== media.id);
              }
            }
          });
        }
      });
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.courseSessionMedia.indexOf(row);
  }
}
