import { HttpClient, HttpEvent } from "@angular/common/http";
import { inject, Injectable } from "@angular/core";
import { CourseSessionMedia } from "@core/domain-classes/course-session-media";
import { CourseSessionMediaChunk } from "@core/domain-classes/course-session-media-chunk";
import { CourseSessionMediaChunkDownload } from "@core/domain-classes/course-session-media-chunk-download";
import { CourseSessionMediaChunkStatus } from "@core/domain-classes/course-session-media-chunk-status";
import { MediaPreview } from "@core/domain-classes/media-previw-data";
import { CommonError } from "@core/error-handler/common-error";
import { CommonHttpErrorService } from "@core/error-handler/common-http-error.service";
import { catchError, Observable, of, retry, timer } from "rxjs";

@Injectable({
    providedIn: 'root'
})
export class SessionMediaService {
    maxRetries: number = 2;
    private httpClient = inject(HttpClient);
    private commonHttpErrorService = inject(CommonHttpErrorService);

    constructor() { }

    getSessionMedia(courseSessionId: string): Observable<CourseSessionMedia[] | CommonError> {
        const url = `CourseSessionMedia/${courseSessionId}`;
        return this.httpClient
            .get<CourseSessionMedia[]>(url)
            .pipe(catchError(this.commonHttpErrorService.handleError));
    }


    addChunkDocument(courseSessionMedia: CourseSessionMedia): Observable<CourseSessionMedia | CommonError> {
        const url = `CourseSessionMedia/chunk`;
        return this.httpClient
            .post<CourseSessionMedia>(url, courseSessionMedia)
            .pipe(catchError(this.commonHttpErrorService.handleError));
    }

    uploadChunkDocument(documentChunkForm: FormData): Observable<CourseSessionMediaChunk | CommonError> {
        const url = `CourseSessionMedia/chunk/upload`;
        return this.httpClient
            .post<CourseSessionMediaChunk>(url, documentChunkForm)
            .pipe(
                retry({
                    count: this.maxRetries, // Retry up to 2 times
                    delay: (error, retryCount) => {
                        console.warn(`Retrying chunk... Attempt ${retryCount}`);
                        return timer(Math.pow(2, retryCount) * 1000); // Exponential backoff: 1s, 2s, 4s...
                    }
                })
            );
    }

    markChunkAsUploaded(courseSessionMediaId?: string, status?: boolean): Observable<CourseSessionMediaChunkStatus> {
        const url = `CourseSessionMedia/chunk/uploadStatus`;
        return this.httpClient.post<CourseSessionMediaChunkStatus>(url, { courseSessionMediaId, status });
    }

    saveIndividualDocument(
        document: CourseSessionMedia
    ): Observable<CourseSessionMedia | CommonError> {
        const url = `CourseSessionMedia`;
        const formData = new FormData();
        if (document.file) {
            formData.append('files', document.file as Blob);
        }
        formData.append('name', document.name);
        formData.append('storageSettingId', document.storageSettingId ?? '');
        formData.append('description', document.description ?? '');
        formData.append('extension', document.extension ?? '');
        formData.append('courseSessionId', document.courseSessionId);
        formData.append('mediaType', document.mediaType ? document.mediaType.toString() : '0');
        formData.append('url', document.url);
        return this.httpClient
            .post<CourseSessionMedia>(url, formData)
            .pipe(catchError(this.commonHttpErrorService.handleError));
    }

    deleteSessionMedia(id: string): Observable<boolean | CommonError> {
        const url = `CourseSessionMedia/${id}`;
        return this.httpClient
            .delete<boolean>(url)
            .pipe(catchError(this.commonHttpErrorService.handleError));
    }

    downloadDocumentChunk(sessionMediaId: string, chunkIndex: number): Observable<CourseSessionMediaChunkDownload | CommonError> {
        const url = `CourseSessionMedia/${sessionMediaId}/chunk/${chunkIndex}/download`;
        return this.httpClient.get<CourseSessionMediaChunkDownload>(
            url
        ).pipe(
            retry({
                count: this.maxRetries, // Retry up to 2 times
                delay: (error, retryCount) => {
                    console.warn(`Retrying chunk... Attempt ${retryCount}`);
                    return timer(Math.pow(2, retryCount) * 1000); // Exponential backoff: 1s, 2s, 4s...
                }
            })
        );
    }

    getDocumentChunks(sessionMediaId: string): Observable<CourseSessionMediaChunk[]> {
        const url = `CourseSessionMedia/${sessionMediaId}/chunks`;
        return this.httpClient.get<CourseSessionMediaChunk[]>(
            url
        );
    }

    downloadDocument(
        sessionMedia?: MediaPreview
    ): Observable<HttpEvent<Blob> | CommonError> {
        let url = `CourseSessionMedia/${sessionMedia?.id}/download`;
        return this.httpClient
            .get(url, {
                reportProgress: true,
                observe: 'events',
                responseType: 'blob',
            })
            .pipe(
                retry({
                    count: this.maxRetries, // Retry up to 2 times
                    delay: (error, retryCount) => {
                        console.warn(`Retrying chunk... Attempt ${retryCount}`);
                        return timer(Math.pow(2, retryCount) * 1000); // Exponential backoff: 1s, 2s, 4s...
                    }
                }),
                catchError((error) =>
                    this.commonHttpErrorService.handleError(
                        this.blobToString(error.error)
                    )
                )
            );
    }

    checkDocumentStoreAsChunk(sessionMediaId: string): Observable<boolean> {
        const url = `CourseSessionMedia/${sessionMediaId}/isChunked`;
        return this.httpClient.get<boolean>(
            url
        );
    }

    getDocumentToken(
        courseSessionMedia?: MediaPreview
    ): Observable<{ [key: string]: string }> {
        let url = `CourseSessionMediaToken/${courseSessionMedia?.id}/token`;
        return this.httpClient.get<{ [key: string]: string }>(url);
    }

    deleteDocumentToken(token: string): Observable<boolean> {
        const url = `CourseSessionMediaToken/${token}`;
        return this.httpClient.delete<boolean>(url);
    }

    readDocument(
        documentView?: MediaPreview
    ): Observable<{ [key: string]: string[] } | CommonError> {
        let url = `CourseSessionMedia/${documentView?.id}/readText`;
        return this.httpClient.get<{ [key: string]: string[] }>(url);
    }


    private blobToString(blob: Blob) {
        const url = URL.createObjectURL(blob);
        const xmlRequest = new XMLHttpRequest();
        xmlRequest.open('GET', url, false);
        xmlRequest.send();
        URL.revokeObjectURL(url);
        return JSON.parse(xmlRequest.responseText);
    }

}