import { Component, Inject, inject } from '@angular/core';
import { SessionService } from '../../course-session.service';
import { ToastrService } from '@core/services/toastr-service';
import { MatDialogRef, MAT_DIALOG_DATA, MatDialogModule } from '@angular/material/dialog';
import { BaseComponent } from '../../../../base.component';
import { MatIconModule } from '@angular/material/icon';
import { MatButtonModule } from '@angular/material/button';
import { MatFormFieldModule } from '@angular/material/form-field';
import { MatInputModule } from '@angular/material/input';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { TranslateModule } from '@ngx-translate/core';
import { ReactiveFormsModule, FormBuilder, FormGroup, Validators } from '@angular/forms';

@Component({
  selector: 'app-add-passing-score',
  imports: [
    MatDialogModule,
    MatIconModule,
    MatButtonModule,
    MatFormFieldModule,
    MatInputModule,
    HasClaimDirective,
    TranslateModule,
    ReactiveFormsModule
  ],
  templateUrl: './add-passing-score.html',
  styleUrl: './add-passing-score.scss'
})
export class AddPassingScore extends BaseComponent {
  passingScoreForm: FormGroup;
  fb = inject(FormBuilder);
  sessionService = inject(SessionService);
  toastr = inject(ToastrService);

  get passingScoreControl() {
    return this.passingScoreForm.get('passingScore');
  }

  get passingScorePercentage(): number {
    const score = this.passingScoreForm.get('passingScore')?.value || 0;
    return Math.round((score / this.data.totalQuestions) * 100);
  }

  constructor(
    public dialogRef: MatDialogRef<AddPassingScore>,
    @Inject(MAT_DIALOG_DATA) public data: { sessionId: string, totalQuestions: number },
  ) {
    super();
    this.passingScoreForm = this.fb.group({
      passingScore: [0, [
        Validators.required,
        Validators.min(1),
        Validators.max(this.data.totalQuestions)
      ]]
    });
  }

  updatePassingScore() {
    if (this.passingScoreForm.valid) {
      const passingScore = this.passingScoreForm.get('passingScore')?.value;

      this.sub$.sink = this.sessionService.updatePassingScore(this.data.sessionId, passingScore).subscribe({
        next: () => {
          this.toastr.success(this.translationService.getValue('PASSING_SCORE_UPDATED_SUCCESSFULLY'));
          this.dialogRef.close(true);
        }
      });
    }
  }

  onNoClick(): void {
    this.dialogRef.close(false);
  }
}
