import { Component, Inject, inject, OnInit } from '@angular/core';
import { MAT_DIALOG_DATA, MatDialogModule, MatDialogRef } from '@angular/material/dialog';
import { TranslateModule } from '@ngx-translate/core';
import { FormArray, FormBuilder, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { SessionService } from '../../course-session.service';

import { ToastrService } from '@core/services/toastr-service';
import { QuizQuestion, QuizQuestionOption } from '@core/domain-classes/session-quiz-question';
import { CommonError } from '@core/error-handler/common-error';
import { MatCheckboxModule } from '@angular/material/checkbox';
import { BaseComponent } from '../../../../base.component';
import { MatButtonModule } from '@angular/material/button';
import { MatIconModule } from '@angular/material/icon';

@Component({
  selector: 'app-add-quiz',
  imports: [
    TranslateModule,
    MatDialogModule,
    ReactiveFormsModule,
    MatCheckboxModule,
    MatButtonModule,
    MatIconModule
  ],
  templateUrl: './add-quiz-question.component.html',
  styleUrl: './add-quiz-question.component.scss'
})
export class AddQuizComponent extends BaseComponent implements OnInit {
  quizQuestionForm!: FormGroup;
  fb = inject(FormBuilder);
  sessionService = inject(SessionService)
  toastr = inject(ToastrService);
  isEdit = false;

  constructor(
    public dialogRef: MatDialogRef<AddQuizComponent>,
    @Inject(MAT_DIALOG_DATA) public data: { sessionId: string, question: QuizQuestion },
  ) {
    super();
  }

  ngOnInit(): void {
    this.creatQuizForm();
    if (!this.data.question) {
      this.addOption();
      this.addOption();
      this.quizQuestionForm.get('courseSessionId')?.setValue(this.data.sessionId);
    } else {
      this.isEdit = true;
      this.quizQuestionForm.get('courseSessionId')?.setValue(this.data.sessionId);
      this.quizQuestionForm.get('id')?.setValue(this.data.question.id)
      this.quizQuestionForm.get('questionText')?.setValue(this.data.question.questionText)
      this.data.question.quizQuestionOptions.forEach((option) => {
        const creatOption = this.fb.group({
          id: [option.id],
          optionText: [option.optionText],
          quizQuestionId: [option.quizQuestionId],
          isCorrect: [option.isCorrect]
        })
        this.optionArray.push(creatOption);
      });
    }
  }

  get optionArray() {
    return this.quizQuestionForm.get('quizQuestionOptions') as FormArray
  }

  creatQuizForm() {
    this.quizQuestionForm = this.fb.group({
      id: [''],
      courseSessionId: [''],
      orderNo: [0],
      questionText: ['', Validators.required],
      quizQuestionOptions: this.fb.array([]),
    })
  }

  addOption() {
    this.optionArray.push(this.creatOption());
  }

  creatOption() {
    return this.fb.group({
      id: [''],
      optionText: ['', Validators.required],
      quizQuestionId: [''],
      isCorrect: [false]
    })
  }

  removeOption(index: number) {
    this.optionArray.removeAt(index);
  }

  onNoClick(): void {
    this.dialogRef.close();
  }

  onSubmit() {
    if (!this.quizQuestionForm.valid) {
      this.quizQuestionForm.markAllAsTouched();
      return;
    }

    const options: QuizQuestionOption[] = this.optionArray.value;
    const hasCorrectOption = options.some(option => option.isCorrect);

    if (!hasCorrectOption) {
      this.toastr.error(this.translationService.getValue('AT_LEAST_ONE_CORRECT_OPTION_REQUIRED'));
      return;
    }

    if (!this.isEdit) {
      this.sessionService.addSessionQuizQuestion(this.quizQuestionForm.getRawValue()).subscribe({
        next: (question: QuizQuestion | CommonError) => {
          this.toastr.success(this.translationService.getValue('QUESTION_SAVE_SUCCESSFULLY'));
          const questionResponse = question as QuizQuestion;
          this.dialogRef.close(questionResponse);
        },
        error: (err) => {
          this.toastr.error(err.error);
        }
      })
    } else if (this.isEdit) {
      this.sessionService.updateQuizQuestion(this.quizQuestionForm.getRawValue()).subscribe({
        next: (question: QuizQuestion | CommonError) => {
          this.toastr.success(this.translationService.getValue('QUESTION_UPDATE_SUCCESSFULLY'));
          const questionResponse = question as QuizQuestion;
          this.dialogRef.close(questionResponse);
        },
        error: (err) => {
          this.toastr.error(err.error);
        }
      })
    }
  }
}
