import { Component, inject, OnInit } from '@angular/core';
import { TranslateModule } from '@ngx-translate/core';
import { MatDialog, MatDialogModule } from '@angular/material/dialog';
import { AddQuizComponent } from './add-quiz-question/add-quiz-question.component';
import { SessionService } from '../course-session.service';
import { ActivatedRoute, Router, RouterLink } from '@angular/router';
import { NgClass } from '@angular/common';
import { SessionQuiz } from '@core/domain-classes/session-quiz';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';

import { ToastrService } from '@core/services/toastr-service';
import { QuizQuestion } from '@core/domain-classes/session-quiz-question';
import { MatCheckboxModule } from '@angular/material/checkbox';
import { BaseComponent } from '../../../base.component';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { MatButtonModule } from '@angular/material/button';
import { MatIconModule } from '@angular/material/icon';
import { AddPassingScore } from './add-passing-score/add-passing-score';

@Component({
  selector: 'app-session-quiz',
  imports: [
    TranslateModule,
    MatCheckboxModule,
    MatDialogModule,
    PageHelpTextComponent,
    HasClaimDirective,
    MatButtonModule,
    NgClass,
    RouterLink,
    MatIconModule
  ],
  templateUrl: './session-quiz.component.html',
  styleUrls: ['./session-quiz.component.scss']
})
export class SessionQuizComponent extends BaseComponent implements OnInit {
  dialog = inject(MatDialog);
  sessionService = inject(SessionService);
  toastr = inject(ToastrService);
  route = inject(ActivatedRoute);
  router = inject(Router);
  sessionQuiz: SessionQuiz = {
    id: '',
    courseSessionId: '',
    passingScore: 0,
    quizQuestions: []
  }
  commandDialogService = inject(CommonDialogService);
  sessionId = '';
  courseId = '';

  ngOnInit(): void {
    this.route.parent?.params.subscribe(params => {
      this.sessionId = params['sessionId'];
      this.courseId = params['courseId'];
    })

    this.sessionService.getSessionQuizQuestion(this.sessionId).subscribe({
      next: (resp) => {
        const sessionQuiz = resp as SessionQuiz
        if (sessionQuiz) {
          this.sessionQuiz = sessionQuiz
        } else {
          this.sessionQuiz = {
            id: '',
            courseSessionId: this.sessionId,
            passingScore: 0,
            quizQuestions: []
          }
        }
      }
    })
  }

  addQuizQuestion(): void {
    const dialogRef = this.dialog.open(AddQuizComponent, {
      width: '700px',
      maxHeight: '90vh',
      data: Object.assign({}, {
        sessionId: this.sessionId,
      })
    });
    dialogRef.afterClosed().subscribe((data: QuizQuestion) => {
      if (data) {
        this.sessionQuiz.quizQuestions.push(data);
      }
    });
  }

  editQuizQuestion(question: QuizQuestion) {
    const dialogRef = this.dialog.open(AddQuizComponent, {
      maxWidth: '50vw',
      width: '100%',
      maxHeight: '90vh',
      data: Object.assign({}, {
        sessionId: this.sessionId,
        question: question
      })
    });
    dialogRef.afterClosed().subscribe((data: QuizQuestion) => {
      if (data) {
        const index = this.sessionQuiz.quizQuestions.findIndex(d => d.id == data.id);
        if (index >= 0) {
          this.sessionQuiz.quizQuestions[index] = data
        }
      }
    })
  }

  deleteQuestion(questionId: string) {
    this.commandDialogService.deleteConfirmtionDialog(this.translationService.getValue("ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_SESSION")).subscribe({
      next: (result: boolean) => {
        if (result) {
          this.sub$.sink = this.sessionService.deleteQuizQuestion(questionId).subscribe({
            next: () => {
              this.sessionQuiz.quizQuestions = this.sessionQuiz.quizQuestions.filter((question) => question.id != questionId);
              this.toastr.success(this.translationService.getValue('QUESTION_DELETE_SUCCESSFULLY'))
            },
            error: (err) => {
              this.toastr.error(err.error);
            }
          })
        }
      }
    });
  }

  managePassingScore() {
    const dialogRef = this.dialog.open(AddPassingScore, {
      width: '100%',
      maxWidth: '500px',
      data: Object.assign({}, {
        sessionId: this.sessionId,
        totalQuestions: this.sessionQuiz.quizQuestions.length,
      })
    });

    dialogRef.afterClosed().subscribe((data: boolean) => {
      if (data) {
        this.router.navigate(['/course/manage', this.courseId]);
      }
    });
  }
}
