import { Session } from "@core/domain-classes/session"
import { patchState, signalStore, withHooks, withMethods, withState } from "@ngrx/signals"
import { SessionService } from "../course-session.service"
import { inject } from "@angular/core"
import { ToastrService } from '@core/services/toastr-service';
import { TranslationService } from "@core/services/translation.service"
import { SessionResource } from "@core/domain-classes/session-resource"


type SessionState = {
    sessionList: Session[],
    curruntSession: Session,
    isAddorUpdate: boolean,
    loadList: boolean,
    filterData: SessionResource
}

const initialState: SessionState = {
    sessionList: [],
    isAddorUpdate: false,
    loadList: true,
    filterData: {
        title: '',
        orderBy: 'title asc',
        pageSize: 10,
        skip: 0,
        totalCount: 0
    },
    curruntSession: {
        id: '',
        courseId: '',
        title: '',
        description: ''
    }
}

export const SessionStore = signalStore(
    { providedIn: 'root' },
    withState(initialState),
    withMethods((
        store,
        sessionService = inject(SessionService),
        toastr = inject(ToastrService),
        translationService = inject(TranslationService)
    ) => ({
        getCourseSession(courseId: string) {
            sessionService.getCourseSession(courseId).subscribe({
                next: (resp) => {
                    const sessions = resp as Session[];
                    if (sessions) {
                        patchState(store, { sessionList: [...sessions], loadList: false });
                    }
                }
            })
        },
        addCourseSession(session: Session) {
            sessionService.addSession(session).subscribe({
                next: (resp) => {
                    const session = resp as Session
                    patchState(store, { curruntSession: { ...session }, isAddorUpdate: true, loadList: true }),
                        toastr.success(translationService.getValue('SESSION_SAVE_SUCCESSFULLY'))
                },
                error: (err) => {
                    toastr.error(err.error);
                }
            })
        },
        updateCourseSession(updateSession: Session) {
            sessionService.updateSession(updateSession).subscribe({
                next: (resp) => {
                    const session = resp as Session
                    patchState(store, {
                        sessionList: store.sessionList().map((session) =>
                            updateSession.id === session.id ? updateSession : session),
                        curruntSession: { ...session }, isAddorUpdate: true
                    }),
                        toastr.success(translationService.getValue('SESSION_SAVE_SUCCESSFULLY'))
                },
                error: (err) => {
                    toastr.error(err.error);
                }
            })
        },
        deleteCourseSession(sessionId: string) {
            sessionService.deleteSession(sessionId).subscribe({
                next: () => {
                    patchState(store, { sessionList: store.sessionList().filter(d => d.id != sessionId), loadList: true })
                    toastr.success(translationService.getValue('SESSION_DELETE_SUCCESSFULLY'))
                },
            })
        },
        resetIsAddFlag() {
            patchState(store, { isAddorUpdate: false })
        },
        setCurrentSession(session: Session) {
            patchState(store, { curruntSession: { ...session } })
        }
    })
    ),
    withHooks({
        onInit() { }
    })
);
