import { AfterViewInit, Component, inject, OnInit, ViewChild } from '@angular/core';
import { MatPaginator } from '@angular/material/paginator';
import { MatSelectModule } from '@angular/material/select';
import { MatTableModule } from '@angular/material/table';
import { IdName } from '@core/domain-classes/id-name';
import { PageSizeOption } from '@core/utils/global-config';
import { TranslateModule } from '@ngx-translate/core';
import { debounceTime, distinctUntilChanged, merge, Subject, switchMap, tap } from 'rxjs';
import { BaseComponent } from '../../base.component';
import { FormControl, FormsModule, ReactiveFormsModule } from '@angular/forms';
import { MatDatepickerModule } from '@angular/material/datepicker';
import { MatInputModule } from '@angular/material/input';
import { MatNativeDateModule } from '@angular/material/core';
import { MatSort, MatSortModule } from '@angular/material/sort';
import { ResponseHeader } from '@core/domain-classes/response-header';
import { MatDialog } from '@angular/material/dialog';
import { CourseService } from '../course/course.service';
import { MatDivider } from '@angular/material/divider';
import { Session } from '@core/domain-classes/session';
import { SessionStatus } from '@core/domain-classes/session-status';
import { HttpResponse } from '@angular/common/http';
import { SessionService } from '../course-session/course-session.service';
import { NgClass } from '@angular/common';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { EmployeeCourseResource } from '@core/domain-classes/employee-course-resource';
import { CourseEmployee } from '@core/domain-classes/course-employee';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';
import { EmployeeCourseStatusPipe } from '../pipes/employee-course-status.pipe';
import { EmployeeCourseStatus } from '../employee-training/employee-course-status.enum';
import { EmployeeAssignCourseCount } from '@core/domain-classes/employee-assign-course-count';
import { ActivatedRoute } from '@angular/router';

@Component({
  selector: 'app-course-summary',
  imports: [
    TranslateModule,
    MatTableModule,
    MatSelectModule,
    MatSortModule,
    ReactiveFormsModule,
    FormsModule,
    MatDatepickerModule,
    MatInputModule,
    MatNativeDateModule,
    MatDivider,
    MatPaginator,
    PageHelpTextComponent,
    UTCToLocalTime,
    MatIconModule,
    MatCardModule,
    NgClass,
    EmployeeCourseStatusPipe
  ],
  templateUrl: './course-summary.component.html',
  styleUrl: './course-summary.component.css'
})
export class CourseSummaryComponent
  extends BaseComponent
  implements OnInit, AfterViewInit {
  pageOption = PageSizeOption;
  isSuperAdmin = false;
  displayedColumns = ['employeeName', 'assignedDate', 'status', 'completedDate'];
  displayedColumnsecond = ['name-search', 'assignedDate-search', 'status-search', 'completedDate-search'];
  footerToDisplayed = ['footer'];
  courseService = inject(CourseService);
  sessionService = inject(SessionService);
  courseNameControl = new FormControl();
  @ViewChild(MatPaginator) paginator!: MatPaginator;
  @ViewChild(MatSort) sort!: MatSort;
  employeeSearchParameter: EmployeeCourseResource = {
    employeeName: '',
    employeeNo: '',
    courseId: '',
    status: null,
    skip: 0,
    pageSize: 10,
    totalCount: 0,
    orderBy: 'createdDate desc',
  };
  filterParameter$: Subject<string> = new Subject<string>();
  _filterEmployeeName = this.employeeSearchParameter.employeeName;
  _filterCourseId = this.employeeSearchParameter.courseId;
  _filterStatus = this.employeeSearchParameter.status;
  status = SessionStatus;
  courseList: IdName[] = [];
  sessionList: Session[] = [];
  dataSource: CourseEmployee[] = [];
  employeeAssignCourseCount = {} as EmployeeAssignCourseCount;
  dialog = inject(MatDialog);
  route = inject(ActivatedRoute);

  statusList = Object.keys(EmployeeCourseStatus)
    .filter((key) => !isNaN(Number(EmployeeCourseStatus[key as any])))
    .map((key) => ({
      label: key,
      value: EmployeeCourseStatus[key as keyof typeof EmployeeCourseStatus],
    }));

  get courseFilter() {
    return this._filterCourseId;
  }

  set courseFilter(courseId: string) {
    if (this._filterCourseId != courseId) {
      this._filterCourseId = courseId;
      const filterCourseId = `filterCourseId#${courseId}`;
      this.filterParameter$.next(filterCourseId);
      this.getAssignCourseCount(courseId);
    }
  }

  get employeeNameFilter() {
    return this._filterEmployeeName;
  }

  set employeeNameFilter(character: string) {
    if (this._filterEmployeeName != character) {
      this._filterEmployeeName = character;
      const filterEmployeeName = `filterEmployeeName#${character}`;
      this.filterParameter$.next(filterEmployeeName);
    }
  }

  get statusFilter() {
    return this._filterStatus;
  }

  set statusFilter(status: number | null) {
    if (this._filterStatus !== status) {
      this._filterStatus = status;
      const filterStatus = `filterStatus#${status}`;
      this.filterParameter$.next(filterStatus);
    }
  }

  ngOnInit(): void {
    this.route.paramMap.subscribe(params => {
      const courseId = params.get('id') ?? '';
      if (courseId) {
        this.courseFilter = courseId;
        this.employeeSearchParameter.courseId = courseId;
        this.loadCourseEmployees(this.employeeSearchParameter);
      }
    });
    this.courseNameChangeValue();
    this.courseNameControl.setValue('');
    this.sub$.sink = this.filterParameter$
      .pipe(debounceTime(1000), distinctUntilChanged())
      .subscribe((c: string) => {
        this.employeeSearchParameter.skip = 0;
        this.paginator.pageIndex = 0;
        const filterArray: Array<string> = c.split('#');
        if (filterArray[0] === 'filterEmployeeName') {
          this.employeeSearchParameter.employeeName = filterArray[1];
        } else if (filterArray[0] === 'filterCourseId') {
          this.employeeSearchParameter.courseId = filterArray[1];
        } else if (filterArray[0] === 'filterStatus') {
          this.employeeSearchParameter.status = filterArray[1] ? Number(filterArray[1]) : null;
        }
        this.loadCourseEmployees(this.employeeSearchParameter);
      });
  }

  getAssignCourseCount(courseId: string) {
    if (!courseId) {
      this.employeeAssignCourseCount = {} as EmployeeAssignCourseCount;
      return;
    }

    this.sub$.sink = this.courseService.getEmployeeCourseStatusCount(courseId).subscribe({
      next: (response) => {
        const count = response as EmployeeAssignCourseCount;
        if (count) {
          this.employeeAssignCourseCount = count;
        }
      },
    });
  }

  courseNameChangeValue() {
    this.sub$.sink = this.courseNameControl.valueChanges
      .pipe(
        debounceTime(500),
        distinctUntilChanged(),
        switchMap((c) => {
          return this.courseService.getCourseForDropdownInput(c);
        })
      )
      .subscribe((resp) => {
        if (resp) {
          const courses = resp as HttpResponse<IdName[]>;
          if (courses && courses.body) {
            this.courseList = [...courses.body];
          }
        }
      })
  }

  loadCourseEmployees(resource: EmployeeCourseResource) {
    this.courseService
      .getCourseEmployees(resource)
      .subscribe({
        next: (httpResponse) => {
          const response = httpResponse as HttpResponse<CourseEmployee[]>;
          if (response && response.body) {
            this.dataSource = [...response.body];
          }
          if (response && response.headers.get('X-Pagination')) {
            const paginationParam = JSON.parse(
              response.headers.get('X-Pagination') ?? ''
            ) as ResponseHeader;
            this.employeeSearchParameter.totalCount =
              paginationParam.totalCount;
          }
        },
        error: (err) => {
          console.error('Error fetching course session employees:', err);
        },
      });
  }

  ngAfterViewInit() {
    this.sub$.sink = this.sort.sortChange.subscribe(() => {
      this.paginator.pageIndex = 0;
    });
    this.sub$.sink = merge(this.sort.sortChange, this.paginator.page)
      .pipe(
        tap(() => {
          this.employeeSearchParameter.skip =
            this.paginator.pageIndex * this.paginator.pageSize;
          this.employeeSearchParameter.pageSize = this.paginator.pageSize;
          this.employeeSearchParameter.orderBy =
            this.sort.active + ' ' + this.sort.direction;
        })
      )
      .subscribe(() => {
        if (this.employeeSearchParameter.courseId) {
          this.loadCourseEmployees(this.employeeSearchParameter);
        }
      });
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.dataSource.indexOf(row);
  }
}