import { SelectionModel } from '@angular/cdk/collections';
import { Component, inject, OnInit, ViewChild } from '@angular/core';
import { FormGroup, FormBuilder, FormControl, FormArray, Validators, ReactiveFormsModule, FormsModule } from '@angular/forms';
import { MatCheckboxModule } from '@angular/material/checkbox';
import { MatPaginator, MatPaginatorModule } from '@angular/material/paginator';
import { MatSort, MatSortModule } from '@angular/material/sort';
import { MatTableModule } from '@angular/material/table';
import { ActivatedRoute, Router, RouterLink } from '@angular/router';
import { AssignCourseCompanyRequest } from '@core/domain-classes/assign-course-company-request';
import { ResponseHeader } from '@core/domain-classes/document-header';
import { Employee } from '@core/domain-classes/employee';
import { EmployeeResource } from '@core/domain-classes/employee-resource';
import { IdName } from '@core/domain-classes/id-name';
import { SecurityService } from '@core/security/security.service';
import { TranslateModule } from '@ngx-translate/core';

import { ToastrService } from '@core/services/toastr-service';
import { Subject, debounceTime, distinctUntilChanged, merge, tap, switchMap } from 'rxjs';
import { BaseComponent } from '../../../base.component';
import { CourseService } from '../course.service';
import { MatSelectModule } from '@angular/material/select';
import { MatDividerModule } from '@angular/material/divider';
import { HttpResponse } from '@angular/common/http';
import { PageSizeOption } from '@core/utils/global-config';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { MatButtonModule } from '@angular/material/button';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';
import { NgClass } from '@angular/common';

@Component({
  selector: 'app-assign-course',
  imports: [
    TranslateModule,
    MatCheckboxModule,
    MatTableModule,
    MatSortModule,
    MatPaginatorModule,
    ReactiveFormsModule,
    MatSelectModule,
    MatDividerModule,
    RouterLink,
    PageHelpTextComponent,
    FormsModule,
    MatButtonModule,
    MatIconModule,
    MatCardModule,
    NgClass
  ],
  templateUrl: './assign-course.component.html',
  styleUrl: './assign-course.component.scss'
})
export class AssignCourseComponent extends BaseComponent implements OnInit {
  displayedColumns: string[] = ['action', 'firstName', 'lastName', 'email'];
  displayedColumnSecond: string[] = ['action-search', 'firstName-search', 'lastName-search', 'email-search'];
  footerToDisplayed = ['footer'];
  courseList: IdName[] = [];
  route = inject(ActivatedRoute);
  @ViewChild(MatPaginator) paginator!: MatPaginator;
  @ViewChild(MatSort) set sort(sort: MatSort) {
    if (sort) {
      this.subscribeSortChanges(sort);
    }
  }
  router = inject(Router);
  employeeSearchParameter: EmployeeResource = {
    name: '',
    employeeNo: '',
    email: '',
    firstName: '',
    lastName: '',
    courseId: '',
    skip: 0,
    pageSize: 10,
    totalCount: 0,
    orderBy: ''
  }
  filterParameter$: Subject<string> = new Subject<string>();
  _filterEmployeeNo = this.employeeSearchParameter.employeeNo;
  _filterFirstName = this.employeeSearchParameter.firstName;
  _filterLastName = this.employeeSearchParameter.lastName;
  _filterEmail = this.employeeSearchParameter.email;
  companyCourseForm!: FormGroup;
  fb = inject(FormBuilder);
  selection = new SelectionModel<Employee>(true, []);
  securityService = inject(SecurityService);
  isSuerAdmin = false;
  searchCourseControl = new FormControl();
  toastr = inject(ToastrService);
  dataSource: Employee[] = [];
  isSuperAdmin = false;
  courseService = inject(CourseService);
  pageOption = PageSizeOption;

  get employeeNoFilter() {
    return this._filterEmployeeNo;
  }

  set employeeNoFilter(character: string) {
    if (this._filterEmployeeNo != character) {
      this._filterEmployeeNo = character;
      const filterEmployeeNo = `filterEmployeeNo#${character}`;
      this.filterParameter$.next(filterEmployeeNo);
    }
  }
  get firstNameFilter() {
    return this._filterFirstName;
  }

  set firstNameFilter(character: string) {
    if (this._filterFirstName != character) {
      this._filterFirstName = character;
      const filtername = `filterFirstName#${character}`;
      this.filterParameter$.next(filtername);
    }
  }

  get lastNameFilter() {
    return this._filterLastName;
  }

  set lastNameFilter(character: string) {
    if (this._filterLastName != character) {
      this._filterLastName = character;
      const filterLastname = `filterLastName#${character}`;
      this.filterParameter$.next(filterLastname);
    }
  }

  get emailFilter() {
    return this._filterEmail;
  }
  set emailFilter(character: string) {
    if (this._filterEmail != character) {
      this._filterEmail = character;
      const filterEmail = `filterEmail#${character}`;
      this.filterParameter$.next(filterEmail);
    }
  }

  get employeeIdArray() {
    return this.companyCourseForm.get('employeeIds') as FormArray;
  }

  constructor() {
    super();
  }

  ngOnInit(): void {
    this.creatAssignCourseForm();
    this.courseNameChangeValue();
    this.searchCourseControl.setValue('');

    this.couseIdSubscription();

    const courseId = this.route.snapshot.queryParamMap.get('courseId');

    this.companyCourseForm.get('courseId')?.setValue(courseId);
    this.employeeSearchParameter.courseId = courseId ?? '';

    this.sub$.sink = this.filterParameter$
      .pipe(
        debounceTime(1000),
        distinctUntilChanged()
      ).subscribe((c: string) => {
        this.employeeSearchParameter.skip = 0;
        this.paginator.pageIndex = 0;
        const filterArray: Array<string> = c.split('#')
        if (filterArray[0] === 'filterEmployeeNo') {
          this.employeeSearchParameter.employeeNo = filterArray[1]
        }
        else if (filterArray[0] === 'filterFirstName') {
          this.employeeSearchParameter.firstName = filterArray[1]
        }
        else if (filterArray[0] === 'filterLastName') {
          this.employeeSearchParameter.lastName = filterArray[1]
        }
        else if (filterArray[0] === 'filterEmail') {
          this.employeeSearchParameter.email = filterArray[1]
        }
        this.loadCompanyCourse();
      });
  }

  couseIdSubscription() {
    this.companyCourseForm.get('courseId')?.valueChanges.subscribe(() => {
      this.selection.clear();
      this.employeeIdArray.clear();
      this.loadCompanyCourse();
    });
  }

  creatAssignCourseForm() {
    this.companyCourseForm = this.fb.group({
      courseId: ['', [Validators.required]],
      isAssignToAllEmployee: [false],
      employeeIds: this.fb.array([]),
    });
  }

  isAllSelected() {
    const numSelected = this.selection.selected.length;
    const numRows = this.dataSource.length;
    return numRows > 0 && numSelected === numRows;
  }

  masterToggle() {
    if (this.isAllSelected()) {
      this.selection.clear();
      this.employeeIdArray.clear();
      this.dataSource.forEach(employee => {
        if (employee.isAssigned) {
          this.selection.select(employee);
          this.employeeIdArray.push(this.fb.control(employee.id));
        }
      });
    } else {
      this.dataSource.forEach(employee => {
        this.selection.select(employee);
        const exists = this.employeeIdArray.controls.some(control => control.value === employee.id);
        if (!exists) {
          this.employeeIdArray.push(this.fb.control(employee.id));
        }
      });
    }
  }

  toggleSelection(employee: Employee) {
    this.selection.toggle(employee);
    const index = this.employeeIdArray.controls.findIndex(control => control.value === employee.id);

    if (this.selection.isSelected(employee)) {
      if (index === -1) {
        this.employeeIdArray.push(this.fb.control(employee.id));
      }
    } else {
      if (index > -1) {
        this.employeeIdArray.removeAt(index);
      }
    }
  }

  onSubmit() {
    if (!this.companyCourseForm.valid) {
      this.companyCourseForm.markAllAsTouched();
      return;
    }

    const formData: AssignCourseCompanyRequest = this.companyCourseForm.getRawValue();
    this.courseService.assignCourseToEmployees(formData).subscribe({
      next: () => {
        this.toastr.success(this.translationService.getValue('COMPANY_COURSE_SAVE_SUCCESSFULLY'))
        this.router.navigate(['/course/list']);
      },
      error: (err) => {
        this.toastr.error(err.error);
      }
    });
  }

  subscribeSortChanges(sort: MatSort): void {
    if (sort) {
      this.sub$.sink = sort.sortChange.subscribe(() => {
        this.paginator.pageIndex = 0
      })

      this.sub$.sink = merge(sort.sortChange, this.paginator.page)
        .pipe(
          tap(() => {
            this.employeeSearchParameter.skip =
              this.paginator.pageIndex * this.paginator.pageSize;
            this.employeeSearchParameter.pageSize = this.paginator.pageSize;
            this.employeeSearchParameter.orderBy = sort.active.split(' ').length == 2 ? sort.active : sort.active + ' ' + sort.direction;
            this.loadCompanyCourse();
          })
        ).subscribe();
    }
  }

  loadCompanyCourse() {
    if (this.companyCourseForm.get('courseId')?.value) {
      this.employeeSearchParameter.courseId = this.companyCourseForm.get('courseId')?.value;
      this.courseService.getEmployees(this.employeeSearchParameter).subscribe({
        next: (httpResponse) => {
          const response = httpResponse as HttpResponse<Employee[]>;
          if (response && response.body) {
            this.dataSource = [...response.body];
            this.selection.clear();
            this.dataSource.forEach((employee) => {
              if (employee.isAssigned) {
                this.employeeIdArray.push(this.fb.control(employee.id));
                this.selection.select(employee);
              }
            })
          }
          if (response && response.headers.get('X-Pagination')) {
            const paginationParam = JSON.parse(response.headers.get('X-Pagination') ?? '') as ResponseHeader;
            this.employeeSearchParameter.totalCount = paginationParam.totalCount
          }
        }
      })
    }
  }

  courseNameChangeValue() {
    this.sub$.sink = this.searchCourseControl.valueChanges.pipe(
      debounceTime(500),
      distinctUntilChanged(),
      switchMap((c) => {
        return this.courseService.getCoursesForDropdown(c ?? '');
      })
    ).subscribe(
      (resp) => {
        this.courseList = resp as IdName[];
      });
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.dataSource.indexOf(row);
  }
}
