import { AfterViewInit, Component, inject, OnInit, ViewChild } from '@angular/core';
import { FormsModule } from '@angular/forms';
import { MatIconModule } from '@angular/material/icon';
import { MatMenuModule } from '@angular/material/menu';
import { MatPaginator } from '@angular/material/paginator';
import { MatSort, MatSortModule } from '@angular/material/sort';
import { RouterLink } from '@angular/router';
import { TranslateModule } from '@ngx-translate/core';
import { CourseStore } from '../store/course.store';
import { PageSizeOption } from '@core/utils/global-config';
import { debounceTime, distinctUntilChanged, merge, tap } from 'rxjs';
import { Subject } from 'rxjs';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { CourseResource } from '@core/domain-classes/course-resource';
import { Course } from '@core/domain-classes/course';
import { Session } from '@core/domain-classes/session';
import { trigger, state, style, transition, animate } from '@angular/animations';
import { CourseSessionComponent } from "./course-session/course-session.component";
import { MatTableModule } from '@angular/material/table';
import { BaseComponent } from '../../../base.component';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { MatCardModule } from '@angular/material/card';
import { MatButtonModule } from '@angular/material/button';
import { NgClass } from '@angular/common';

@Component({
  selector: 'app-course-list',
  imports: [
    TranslateModule,
    MatSortModule,
    MatIconModule,
    MatMenuModule,
    FormsModule,
    MatPaginator,
    RouterLink,
    CourseSessionComponent,
    MatTableModule,
    PageHelpTextComponent,
    HasClaimDirective,
    UTCToLocalTime,
    MatCardModule,
    MatButtonModule,
    NgClass
  ],
  templateUrl: './course-list.component.html',
  styleUrl: './course-list.component.scss',
  animations: [
    trigger('detailExpand', [
      state('collapsed', style({ height: '0px', minHeight: '0' })),
      state('expanded', style({ height: '*' })),
      transition('expanded <=> collapsed', animate('225ms cubic-bezier(0.4, 0.0, 0.2, 1)')),
    ]),
  ]
})

export class CourseListComponent extends BaseComponent implements OnInit, AfterViewInit {
  dataSource = [''];
  footerToDisplayed = ['footer'];
  displayedColumnSecond = ['action-search', 'title-search', 'startDate-search', 'endDate-search'];
  displayedColumns = ['action', 'title', 'startDate', 'endDate'];
  courseStore = inject(CourseStore);
  commandDialogService = inject(CommonDialogService);
  courseSearchParameter: CourseResource = { ...this.courseStore.filterParameter() };
  pageOption = PageSizeOption;
  @ViewChild(MatPaginator) paginator!: MatPaginator;
  @ViewChild(MatSort) sort!: MatSort;
  filterParameter$: Subject<string> = new Subject<string>();
  _filterTitle = this.courseSearchParameter.title;
  courseSession: Session[] = [];
  expandedElement!: Course | null;
  subActionColumnToDisplay = ['action', 'session', 'status'];

  get titleFilter() {
    return this._filterTitle;
  }

  set titleFilter(character: string) {
    if (this._filterTitle != character) {
      this._filterTitle = character;
      const filterTitle = `filterTitle#${character}`;
      this.filterParameter$.next(filterTitle);
    }
  }

  ngOnInit(): void {
    this.sub$.sink = this.filterParameter$
      .pipe(
        debounceTime(1000),
        distinctUntilChanged()
      )
      .subscribe((c: string) => {
        this.courseSearchParameter.skip = 0;
        this.paginator.pageIndex = 0;
        const filterArray: Array<string> = c.split('#')
        if (filterArray[0] === 'filterTitle') {
          this.courseSearchParameter.title = filterArray[1]
        }
        this.courseStore.loadByQuery(this.courseSearchParameter)
      })
  }

  refresh() {
    this.courseStore.loadByQuery(this.courseSearchParameter);
  }

  getCourseSession(course: Course) {
    if (course) {
      this.expandedElement = this.expandedElement === course ? null : course
    }
  }

  deleteCourse(courseId: string) {
    this.commandDialogService.deleteConfirmtionDialog(this.translationService.getValue("ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_COURSE")).subscribe({
      next: (result: boolean) => {
        if (result) {
          this.courseStore.deleteCourse(courseId);
        }
      }
    });
  }

  ngAfterViewInit(): void {
    this.sub$.sink = this.sort.sortChange.subscribe(() => {
      this.paginator.pageIndex = 0
    })
    this.sub$.sink = merge(this.sort.sortChange, this.paginator.page)
      .pipe(
        tap(() => {
          this.courseSearchParameter.skip =
            this.paginator.pageIndex * this.paginator.pageSize;
          this.courseSearchParameter.pageSize = this.paginator.pageSize;
          this.courseSearchParameter.orderBy =
            this.sort.active + ' ' + this.sort.direction;
          this.courseStore.loadByQuery(this.courseSearchParameter)
        })
      ).subscribe()
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.courseStore.courseList().indexOf(row);
  }
}
