import { HttpClient, HttpParams, HttpResponse } from '@angular/common/http';
import { inject, Injectable } from '@angular/core';
import { AssignCourseCompanyRequest } from '@core/domain-classes/assign-course-company-request';
import { Course } from '@core/domain-classes/course';
import { CourseResource } from '@core/domain-classes/course-resource';
import { CourseSessionResource } from '@core/domain-classes/course-session-resource';
import { Employee } from '@core/domain-classes/employee';
import { CourseEmployee } from '@core/domain-classes/course-employee';
import { CourseSession } from '@core/domain-classes/employee-assigned-courses';
import { EmployeeResource } from '@core/domain-classes/employee-resource';
import { IdName } from '@core/domain-classes/id-name';
import { CommonError } from '@core/error-handler/common-error';
import { CommonHttpErrorService } from '@core/error-handler/common-http-error.service';
import { Session } from 'inspector';
import { Observable, catchError } from 'rxjs';
import { EmployeeCourseResource } from '@core/domain-classes/employee-course-resource';
import { EmployeeAssignCourseCount } from '@core/domain-classes/employee-assign-course-count';

@Injectable({
  providedIn: 'root'
})
export class CourseService {
  private httpClient = inject(HttpClient);
  private commonHttpErrorService = inject(CommonHttpErrorService)
  constructor() { }

  getCourses(
    resource: CourseResource
  ): Observable<HttpResponse<Course[]>> {
    const url = 'Course';
    const customParams = new HttpParams()
      .set('PageSize', resource.pageSize)
      .set('Skip', resource.skip)
      .set('Title', resource.title.toString())
      .set('Description', resource.description.toString())
      .set('OrderBy', resource.orderBy)
    return this.httpClient
      .get<Course[]>(url, {
        params: customParams,
        observe: 'response',
      })
  }

  getCourse(courseId: string): Observable<Course | CommonError> {
    const url = `Course/${courseId}`
    return this.httpClient
      .get<Course>(url)
      .pipe(catchError(err => this.commonHttpErrorService.handleError(err)));
  }

  getCourseForDropdownInput(
    query: string
  ): Observable<HttpResponse<IdName[]> | CommonError> {
    const url = `Course/search`
    const customParams = new HttpParams()
      .set('query', query ?? '')
    return this.httpClient.get<IdName[]>(url, {
      params: customParams,
      observe: 'response',
    }).pipe(catchError(this.commonHttpErrorService.handleError));
  }

  addCourse(course: Course): Observable<Course | CommonError> {
    const url = 'Course'
    return this.httpClient.post<Course>(url, course)
      .pipe(catchError(err => this.commonHttpErrorService.handleError(err)));
  }

  updateCourse(course: Course): Observable<Course | CommonError> {
    const url = `Course/${course.id}`
    return this.httpClient.put<Course>(url, course)
      .pipe(catchError(err => this.commonHttpErrorService.handleError(err)));
  }

  deleteCourse(courseId: string): Observable<void | CommonError> {
    const url = `Course/${courseId}`;
    return this.httpClient
      .delete<void>(url)
      .pipe(catchError(err => this.commonHttpErrorService.handleError(err)));
  }

  getCourseSessions(courseId: string): Observable<Session[]> {
    const url = `CourseSession/${courseId}/sessions`
    return this.httpClient
      .get<Session[]>(url)
  }

  getSessionForDropdown(query: string): Observable<IdName[] | CommonError> {
    return this.httpClient.get<IdName[]>(`Session/search?query=${query}`)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getCoursesForDropdown(query: string): Observable<IdName[] | CommonError> {
    return this.httpClient.get<IdName[]>(`Course/search?query=${query}`)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getCourseSesssionEmployees(
    resource: CourseSessionResource
  ): Observable<HttpResponse<CourseSession[]>> {
    const url = 'EmployeeCourseSession';
    const customParams = new HttpParams()
      .set('CourseSessionId', resource.courseSessionId ?? '')
      .set('IsCompleted', resource.isCompleted ? resource.isCompleted.toString() : '')
      .set('PageSize', resource.pageSize)
      .set('Skip', resource.skip)
      .set('EmployeeName', resource.employeeName ? resource.employeeName.toString() : '')
      .set('AssignedStartDate', resource.assignedStartDate ? new Date(resource.assignedStartDate).toISOString() : '')
      .set('AssignedEndDate', resource.assignedEndDate ? new Date(resource.assignedEndDate).toISOString() : '')
      .set('CompletionStartDate', resource.completionStartDate ? new Date(resource.completionStartDate).toISOString() : '')
      .set('CompletionEndDate', resource.completionEndDate ? new Date(resource.completionEndDate).toISOString() : '')
      .set('OrderBy', resource.orderBy);
    return this.httpClient
      .get<CourseSession[]>(url, {
        params: customParams,
        observe: 'response',
      });
  }

  getEmployees(
    resource: EmployeeResource
  ): Observable<HttpResponse<Employee[]>> {
    const url = 'EmployeeCourse';
    const customParams = new HttpParams()
      .set('OrderBy', resource.orderBy)
      .set('PageSize', resource.pageSize)
      .set('Skip', resource.skip)
      .set('FirstName', resource.firstName.toString())
      .set('LastName', resource.lastName.toString())
      .set('Email', resource.email.toString())
      .set('EmployeeNo', resource.employeeNo)
      .set('courseId', resource.courseId ?? '')
    return this.httpClient
      .get<Employee[]>(url, {
        params: customParams,
        observe: 'response',
      })
  }

  assignCourseToEmployees(formData: AssignCourseCompanyRequest): Observable<Course | CommonError> {
    return this.httpClient.post<Course>('EmployeeCourse', formData)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getCourseEmployees(
    resource: EmployeeCourseResource
  ): Observable<HttpResponse<CourseEmployee[]>> {
    const url = `EmployeeCourse/${resource.courseId}/employees`;
    const customParams = new HttpParams()
      .set('OrderBy', resource.orderBy)
      .set('PageSize', resource.pageSize)
      .set('Skip', resource.skip)
      .set('EmployeeName', resource.employeeName.toString())
      .set('EmployeeNo', resource.employeeNo)
      .set('courseId', resource.courseId ?? '')
      .set('Status', resource.status ?? '')
    return this.httpClient
      .get<CourseEmployee[]>(url, {
        params: customParams,
        observe: 'response',
      })
  }

  getEmployeeCourseStatusCount(courseId: string): Observable<EmployeeAssignCourseCount | CommonError> {
    const url = `EmployeeCourse/${courseId}/counts`;
    return this.httpClient.get<EmployeeAssignCourseCount>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }
}
