import { Component, inject, OnInit } from '@angular/core';
import { toObservable } from '@angular/core/rxjs-interop';
import { ReactiveFormsModule, FormGroup, FormBuilder, Validators } from '@angular/forms';
import { MatIconModule } from '@angular/material/icon';
import { MatMenuModule } from '@angular/material/menu';
import { MatTableModule } from '@angular/material/table';
import { RouterLink, ActivatedRoute, Router, Data } from '@angular/router';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { TranslateModule } from '@ngx-translate/core';
import { CourseStore } from '../store/course.store';
import { BaseComponent } from '../../../base.component';
import { SessionStore } from '../../course-session/store/course-session.store';
import { MatCheckboxModule } from '@angular/material/checkbox';
import { TextEditorComponent } from '@shared/text-editor/text-editor.component';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { MatButtonModule } from '@angular/material/button';
import { MatCardModule } from '@angular/material/card';
import { NgClass } from '@angular/common';

@Component({
  selector: 'app-manage-course',
  imports: [
    TranslateModule,
    ReactiveFormsModule,
    RouterLink,
    MatTableModule,
    MatMenuModule,
    TextEditorComponent,
    MatCheckboxModule,
    PageHelpTextComponent,
    HasClaimDirective,
    MatButtonModule,
    MatCardModule,
    MatIconModule,
    NgClass
  ],
  templateUrl: './manage-course.component.html',
  styleUrl: './manage-course.component.scss'
})
export class ManageCourseComponent extends BaseComponent implements OnInit {
  courseForm!: FormGroup;
  fb = inject(FormBuilder);
  courseStore = inject(CourseStore);
  sessionStore = inject(SessionStore);
  route = inject(ActivatedRoute);
  router = inject(Router);
  isEdit = false;
  displayedColumns = ['action', 'title']
  courseId = '';
  commandDialogService = inject(CommonDialogService);

  constructor() {
    super();
    this.subscribeIsAddorUpdate();
  }

  ngOnInit(): void {
    this.creatCourseForm()
    this.sub$.sink = this.route.data.subscribe((response: Data) => {
      if (response['courseDetail']) {
        this.isEdit = true;
        this.courseForm.patchValue(response['courseDetail']);
        this.courseStore.setCurrentCouse(response['courseDetail']);
      }
    })
    this.sub$.sink = this.route.params.subscribe(params => {
      this.courseId = params['id'];
      if (this.courseId) {
        this.sessionStore.getCourseSession(this.courseId);
      }
    })
  }

  creatCourseForm() {
    this.courseForm = this.fb.group({
      id: [''],
      title: ['', Validators.required],
      description: [''],
      autoAssignToNewEmployee: [false],
    })
  }

  onSubmit() {
    if (!this.courseForm.valid) {
      this.courseForm.markAllAsTouched();
      return;
    }
    if (!this.isEdit) {
      this.courseStore.addCourse(this.courseForm.getRawValue());
    }
    else if (this.isEdit) {
      this.courseStore.updateCourse(this.courseForm.getRawValue());
    }
  }

  deleteSession(sessionId: string) {
    this.commandDialogService.deleteConfirmtionDialog(this.translationService.getValue("ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_SESSION")).subscribe({
      next: (result: boolean) => {
        if (result) {
          this.sessionStore.deleteCourseSession(sessionId);
        }
      }
    });
  }

  subscribeIsAddorUpdate() {
    toObservable(this.courseStore.isAddorUpdate).subscribe((flag) => {
      if (flag) {
        this.router.navigate([`/course/manage/${this.courseStore.currentCourse()?.id}`]);
      }
      this.courseStore.resetflag();
    });
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.sessionStore.sessionList().indexOf(row);
  }
}
